#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/vault_boy/include/secrets.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/wiki_config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/privacy_guard/lib/tools.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>

int main(int argc, char* argv[])
try {
    maps::cmdline::Parser parser;
    auto configPath = parser.string("config").help("path to configuration");
    auto wikiConfigDirPath = parser.string("wiki-config-dir")
            .defaultValue(maps::mrc::common::WIKI_TEMPLATE_CONFIG_DIR)
            .help("path to services config template for wikimap");
    auto wikiConfigPath =
        parser.string("wiki-config")
            .help("path to services config for wikimap");
    auto syslog = parser.string("syslog-tag")
                      .help("redirect log output to syslog with given tag");
    auto dryRun = parser.flag("dry-run").help("do not update data");
    parser.parse(argc, argv);

    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }

    auto cfg = maps::mrc::common::templateConfigFromCmdPath(configPath);

    auto postgres = cfg.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                                       maps::mrc::common::LONG_READ_POOL_ID);
    maps::pgp3utils::PgAdvisoryXactMutex mutex(
        postgres.pool(),
        static_cast<int64_t>(maps::mrc::common::LockId::PrivacyGuard));
    if (!dryRun && !mutex.try_lock()) {
        INFO() << "another process is ongoing";
        return EXIT_SUCCESS;
    }

    auto wikiConfig = wikiConfigPath.defined()
        ? maps::wiki::common::ExtendedXmlDoc(wikiConfigPath)
        : maps::mrc::common::makeWikiConfig(wikiConfigDirPath);
    maps::wiki::common::PoolHolder wikiPool{
        wikiConfig, "long-read", "long-read"};
    auto loader = maps::mrc::object::makeRevisionLoader(
        wikiPool.pool().slaveTransaction());

    maps::mrc::privacy_guard::verify(*loader, postgres.pool(), dryRun);
    if (!dryRun) {
        maps::mrc::privacy_guard::updateMetadata(postgres.pool());
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
