#include "magic_strings.h"
#include "tools.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/for_each_batch.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/parallel_for_each.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/metadata_gateway.h>
#include <maps/libs/geolib/include/spatial_relation.h>

namespace maps::mrc::privacy_guard {

namespace {

object::MrcRegions loadRestrictedRegions(object::Loader& loader)
{
    object::MrcRegions result = loader.loadAllMrcRegions();
    auto it = std::remove_if(result.begin(), result.end(), [](auto& region) {
        return region.type() != object::MrcRegion::Type::Restricted;
    });
    result.erase(it, result.end());
    INFO() << "loaded " << result.size() << " regions";
    return result;
}

auto loadNotSecretFeatures(sql_chemistry::Transaction& txn,
                           const object::RevisionID& revisionID,
                           const geolib3::Polygon2& mercatorGeom)
{
    auto result = db::FeatureGateway(txn).load(
        db::table::Feature::isPublished &&
        db::table::Feature::pos.intersects(mercatorGeom.boundingBox()));
    auto it = std::remove_if(result.begin(), result.end(), [&](auto& feature) {
        return feature.privacy() == db::FeaturePrivacy::Secret ||
               !spatialRelation(mercatorGeom,
                                feature.mercatorPos(),
                                geolib3::SpatialRelation::Intersects);
    });
    result.erase(it, result.end());
    if (!result.empty()) {
        INFO() << "object " << revisionID.objectId() << ", commit "
               << revisionID.commitId() << ", features " << result.size();
    }
    return result;
}

void update(pgpool3::Pool& pool, db::Features& features)
{
    constexpr size_t BATCH_SIZE = 100;
    common::forEachBatch(features, BATCH_SIZE, [&](auto first, auto last) {
        auto txn = pool.masterWriteableTransaction();
        db::FeatureGateway{*txn}.update(db::Features{first, last},
                                        db::UpdateFeatureTxn::Yes);
        txn->commit();
    });
}

}  // namespace

void verify(object::Loader& loader, pgpool3::Pool& pool, bool dryRun)
{
    auto regions = loadRestrictedRegions(loader);
    size_t counter = 0;
    common::parallelForEach(regions.begin(), regions.end(),
        [&](auto& guard, const auto& region) {
            auto features = loadNotSecretFeatures(
                *pool.slaveTransaction(), region.revisionId(), region.geom());
            if (features.empty()) {
                return;
            }
            if (!dryRun) {
                for (auto& feature : features) {
                    feature.setPrivacy(db::FeaturePrivacy::Secret);
                }
                update(pool, features);
            }
            std::lock_guard lock{guard};
            counter += features.size();
        });
    INFO() << "total features " << counter;
}

void updateMetadata(pgpool3::Pool& pool)
{
    auto txn = pool.masterWriteableTransaction();
    mrc::db::MetadataGateway{*txn}.upsertByKey(
        LAST_RUN_TIME,
        chrono::formatSqlDateTime(chrono::TimePoint::clock::now()));
    txn->commit();
    INFO() << "updated metadata";
}

}  // namespace maps::mrc::privacy_guard
