#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/mock_loader.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/privacy_guard/lib/tools.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/vector.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

namespace maps::mrc::privacy_guard::tests {

using namespace testing;

namespace {

const auto BOX = geolib3::BoundingBox{{0., 0.}, {4., 4.}};
const auto OFFSET = geolib3::Vector2{1., 1.};

auto makeFeature(const geolib3::Point2& pos, db::FeaturePrivacy privacy)
{
    return db::Feature{
        std::string{"sourceId"},
        pos,
        geolib3::Heading{42},
        chrono::formatSqlDateTime(std::chrono::system_clock::now()),
        mds::Key{"groupId", "path"},
        db::Dataset::Agents}
        .setSize(6, 9)
        .setAutomaticShouldBePublished(true)
        .setIsPublished(true)
        .setPrivacy(privacy);
}

}  // namespace

    TEST(suite, test_verify)
    {
        unittest::WithUnittestConfig<unittest::DatabaseFixture> fixture;
        {
            db::Features features{makeFeature(BOX.lowerCorner() + OFFSET,
                                              db::FeaturePrivacy::Secret),
                                  makeFeature(BOX.upperCorner() - OFFSET,
                                              db::FeaturePrivacy::Public),
                                  makeFeature(BOX.upperCorner() + OFFSET,
                                              db::FeaturePrivacy::Public)};
            auto txn = fixture.pool().masterWriteableTransaction();
            db::FeatureGateway{*txn}.insert(features);
            txn->commit();
        }

        object::MockLoader loader;
        loader.add(object::MrcRegions{
            object::MrcRegion(object::RevisionID{1, 1},
                              geolib3::convertGeodeticToMercator(BOX.polygon()))
                .type(object::MrcRegion::Type::Restricted)});

        verify(loader, fixture.pool(), false /*dryRun*/);

        auto features =
            db::FeatureGateway{*fixture.pool().masterReadOnlyTransaction()}
                .load();
        for (const auto& feature : features) {
            bool isSecret =
                spatialRelation(BOX,
                                feature.geodeticPos(),
                                geolib3::SpatialRelation::Intersects);
            EXPECT_EQ(isSecret,
                      feature.privacy() == db::FeaturePrivacy::Secret);
        }
    }

}  // namespace maps::mrc::privacy_guard::tests
