#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/disqualified_source_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/red_card/lib/db.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

#include <random>

namespace maps::mrc::db {

using introspection::operator==;

std::ostream& operator<<(std::ostream& os, const DisqualifiedSource& that)
{
    os << "DisqualifiedSource{.id=" << that.id()
       << ", .disqType=" << that.disqType() << ", .sourceId=" << that.sourceId()
       << ", .startedAt=" << chrono::formatIsoDateTime(that.startedAt());
    if (that.endedAt()) {
        os << ", .endedAt=" << chrono::formatIsoDateTime(*that.endedAt());
    }
    return os << "}";
}

}  // namespace maps::mrc::db

namespace maps::mrc::red_card::tests {

using namespace testing;

const auto SOURCE_ID = std::string{"iPhone"};
const auto NOW = chrono::parseIsoDateTime("2021-02-21T14:43:00+03");
const auto BAD_PROBABILITY = ROAD_PROBABILITY_THRESHOLD - .1;
const auto GOOD_PROBABILITY = ROAD_PROBABILITY_THRESHOLD + .1;

db::Feature makePhoto(double roadProbability)
{
    return sql_chemistry::GatewayAccess<db::Feature>::construct()
        .setDataset(db::Dataset::TaxiSignalQ2)
        .setSourceId(SOURCE_ID)
        .setTimestamp(NOW - RECENT_TIMES / 2)
        .setGeodeticPos(geolib3::Point2{37.5636, 55.6676})
        .setHeading(geolib3::Heading{90})
        .setMdsKey(mds::Key{"a", "b"})
        .setSize({240, 180})
        .setUserId("usr")
        .setRoadProbability(roadProbability);
}

auto& randomNumberEngine()
{
    static auto result = std::mt19937{std::random_device{}()};
    return result;
}

db::Features makeBadPhotos()
{
    ASSERT(REQUIRED_PERCENTILE > .5);
    auto result = db::Features{};
    for (int i = 0; i != REQUIRED_PHOTOS / 2; ++i) {
        result.push_back(makePhoto(BAD_PROBABILITY));
        result.push_back(makePhoto(GOOD_PROBABILITY));
    }
    std::shuffle(result.begin(), result.end(), randomNumberEngine());
    return result;
}

db::Features makeGoodPhotos()
{
    auto result = db::Features{};
    for (int i = 0; i != REQUIRED_PHOTOS; ++i) {
        result.push_back(makePhoto(GOOD_PROBABILITY));
    }
    return result;
}

struct Fixture : testing::Test,
                 unittest::WithUnittestConfig<unittest::DatabaseFixture> {
};

TEST_F(Fixture, test_disable_capturing_bad_photos)
{
    auto disq = db::DisqualifiedSource{db::DisqType::DisableCapturing,
                                       SOURCE_ID,
                                       NOW - RECENT_TIMES,
                                       NOW + RECENT_TIMES};
    auto photos = makeBadPhotos();
    {
        auto txn = pool().masterWriteableTransaction();
        db::DisqualifiedSourceGateway{*txn}.insert(disq);
        db::FeatureGateway{*txn}.insert(photos);
        txn->commit();
    }
    auto result = prepareDisqualifiedSources(
        *pool().masterReadOnlyTransaction(), SOURCE_ID, NOW);
    EXPECT_TRUE(result.empty());
}

TEST_F(Fixture, test_bad_photos)
{
    auto photos = makeBadPhotos();
    {
        auto txn = pool().masterWriteableTransaction();
        db::FeatureGateway{*txn}.insert(photos);
        txn->commit();
    }
    auto result = prepareDisqualifiedSources(
        *pool().masterReadOnlyTransaction(), SOURCE_ID, NOW);
    EXPECT_THAT(result,
                UnorderedElementsAre(
                    db::DisqualifiedSource{db::DisqType::DisableCapturing,
                                           SOURCE_ID,
                                           NOW,
                                           NOW + 2 * RECENT_TIMES},
                    db::DisqualifiedSource{
                        db::DisqType::DisablePublishing, SOURCE_ID, NOW}));
}

TEST_F(Fixture, test_good_photos)
{
    auto photos = makeGoodPhotos();
    {
        auto txn = pool().masterWriteableTransaction();
        db::FeatureGateway{*txn}.insert(photos);
        txn->commit();
    }
    auto result = prepareDisqualifiedSources(
        *pool().masterReadOnlyTransaction(), SOURCE_ID, NOW);
    EXPECT_TRUE(result.empty());
}

TEST_F(Fixture, test_disable_publishing_good_photos)
{
    auto disq = db::DisqualifiedSource{
        db::DisqType::DisablePublishing, SOURCE_ID, NOW - RECENT_TIMES};
    auto photos = makeGoodPhotos();
    {
        auto txn = pool().masterWriteableTransaction();
        db::DisqualifiedSourceGateway{*txn}.insert(disq);
        db::FeatureGateway{*txn}.insert(photos);
        txn->commit();
    }
    auto result = prepareDisqualifiedSources(
        *pool().masterReadOnlyTransaction(), SOURCE_ID, NOW);
    EXPECT_THAT(result, UnorderedElementsAre(disq.setEndedAt(NOW)));
}

}  // namespace maps::mrc::red_card::tests
