#include <maps/wikimap/mapspro/services/mrc/long_tasks/ride_inspector/lib/hypotheses.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/ride_inspector/lib/process_queue.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/queued_photo_id_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/juggler/include/juggler.h>

#include <exception>
#include <iostream>

namespace db = maps::mrc::db;
namespace juggler = maps::mrc::juggler;
namespace ride_inspector = maps::mrc::ride_inspector;

namespace {

constexpr size_t WARN_QUEUE_SIZE{10000};
constexpr size_t ERROR_QUEUE_SIZE{2 * WARN_QUEUE_SIZE};
constexpr std::chrono::hours TIMEOUT{2 * 24};
constexpr size_t WARN_HYPOTHESES_QUEUE_SIZE{1000};
constexpr size_t ERROR_HYPOTHESES_QUEUE_SIZE{2 * WARN_QUEUE_SIZE};

} // anonymous namespace

int main(int argc, char* argv[]) try {
    maps::log8::setLevel(maps::log8::Level::FATAL);
    maps::cmdline::Parser parser("Monitoring for the mrc-ride-inspector");
    auto configPath = parser.string("config").help("path to configuration");
    parser.parse(argc, argv);

    auto cfg = maps::mrc::common::templateConfigFromCmdPath(configPath);
    auto poolHolder = cfg.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                                         maps::mrc::common::LONG_READ_POOL_ID);
    auto txn = poolHolder.pool().slaveTransaction();
    const auto expiredTime = maps::chrono::TimePoint::clock::now() - TIMEOUT;
    auto photos = db::QueuedPhotoIdGateway{*txn}.count(
        db::table::QueuedPhotoId::receivedAt.less(expiredTime));
    auto hypotheses = ride_inspector::hypothesesQueueSize(*txn);
    auto msg = std::ostringstream{} << "expired photos " << photos
                                    << ", unhandled hypotheses " << hypotheses;
    if (photos > ERROR_QUEUE_SIZE || hypotheses > ERROR_HYPOTHESES_QUEUE_SIZE) {
        std::cout << juggler::error(msg.str());
    }
    else if (photos > WARN_QUEUE_SIZE ||
             hypotheses > WARN_HYPOTHESES_QUEUE_SIZE) {
        std::cout << juggler::warn(msg.str());
    }
    else {
        std::cout << juggler::ok(msg.str());
    }
    return EXIT_SUCCESS;
}
catch (const std::exception& e)
{
    std::cout << juggler::error(e.what());
    return EXIT_SUCCESS;
}
catch (...)
{
    std::cout << juggler::error("unknown error");
    return EXIT_SUCCESS;
}
