#include "process_queue.h"
#include "ride.h"
#include "utility.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/collection.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/for_each_passage.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/queued_photo_id_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_gateway.h>
#include <maps/libs/common/include/make_batches.h>

#include <unordered_map>
#include <unordered_set>

namespace maps {
namespace mrc {
namespace ride_inspector {
namespace {

typedef db::Features::iterator FeatureIt;

/// To dequeue the processed photos that have been exported.
db::TIds processedFeatures(Context& ctx, const db::TIds& photoIds)
{
    return db::FeatureGateway{*ctx.pool().masterReadOnlyTransaction()}.loadIds(
        db::table::Feature::id.in(photoIds) &&
        db::table::Feature::processedAt.isNotNull() &&
        (db::table::Feature::isPublished.is(true) ||
         db::table::Feature::shouldBePublished.is(false)));
}

} // anonymous namespace

void processQueue(Context& ctx)
{
    static constexpr size_t BATCH_SIZE = 10000;

    auto queuedPhotoIds = ctx.loadQueuedPhotoIds();
    INFO() << "loaded " << queuedPhotoIds.size() << " queued feature IDs";
    std::sort(queuedPhotoIds.begin(), queuedPhotoIds.end(),
              [](const auto& lhs, const auto& rhs) {
                  return lhs.receivedAt() < rhs.receivedAt();
              });

    size_t processedFeatureCounter = 0;
    size_t completedFeatureCounter = 0;
    for (const auto& batch :
         maps::common::makeBatches(queuedPhotoIds, BATCH_SIZE)) {
        auto photoIds =
            invokeForEach(&db::rides::QueuedPhotoId::photoId, batch);
        auto completedFeatureIds = processedFeatures(ctx, photoIds);
        auto completedFeatureIdsSize = completedFeatureIds.size();
        updateRidesByPhotoIds(ctx.pool(), photoIds);
        if (!completedFeatureIds.empty()) {
            ctx.deleteQueuedPhotoIds(db::table::QueuedPhotoId::photoId.in(
                std::move(completedFeatureIds)));
        }
        processedFeatureCounter += std::distance(batch.begin(), batch.end());
        completedFeatureCounter += completedFeatureIdsSize;
        INFO() << "processed " << processedFeatureCounter << ", completed "
               << completedFeatureCounter;
    }
}

} // ride_inspector
} // mrc
} // maps
