#include "hypotheses.h"
#include "ride.h"
#include "utility.h"

#include <maps/libs/sql_chemistry/include/types.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_utility.h>

namespace maps::mrc::ride_inspector {

namespace {

auto byStart = [](const db::Ride& lhs, const db::Ride& rhs) {
    return lhs.startTime() < rhs.startTime();
};

auto byEnd = [](const db::Ride& lhs, const db::Ride& rhs) {
    return lhs.endTime() < rhs.endTime();
};

auto isDeleted = [](const db::Ride& ride) { return ride.isDeleted(); };

auto getUploadedAt = [](const db::Feature& feature) {
    return feature.uploadedAt();
};

auto isProcessed = [](const db::Feature& feature) {
    if (feature.isPublished()) {
        return true;
    }
    if (!feature.processedAt().has_value()) {
        return false;
    }
    return !feature.shouldBePublished();
};

bool isShowAuthorshipEnabled(const db::Feature& feature)
{
    return feature.showAuthorship().value_or(false);
}

bool isShowAuthorshipEnabled(const db::Ride& ride)
{
    return ride.hasShowAuthorship() && ride.showAuthorship();
}

struct RidesPair {
    std::reference_wrapper<db::Ride> lhs;
    std::reference_wrapper<db::Ride> rhs;

    std::optional<float> iou() const
    {
        auto minStart = std::min(lhs.get().startTime(), rhs.get().startTime());
        auto maxStart = std::max(lhs.get().startTime(), rhs.get().startTime());
        auto minEnd = std::min(lhs.get().endTime(), rhs.get().endTime());
        auto maxEnd = std::max(lhs.get().endTime(), rhs.get().endTime());
        if (maxStart > minEnd) {
            return std::nullopt;
        }
        if (minStart == maxEnd) {
            return 1;
        }
        return (minEnd - maxStart) / (maxEnd - minStart);
    }

    void assignIfPossible()
    {
        if (!lhs.get().isDeleted() || rhs.get().isDeleted() || !iou()) {
            return;
        }
        lhs.get().copyContentFrom(rhs.get());
        rhs.get().setIsDeleted(true);
    }
};

auto makePoints(db::Features::iterator first, db::Features::iterator last)
    -> geolib3::PointsVector
{
    auto result = geolib3::PointsVector{};
    std::for_each(first, last, [&](const auto& photo) {
        if (photo.hasPos()) {
            result.push_back(photo.geodeticPos());
        }
    });
    return result;
}

auto makeTrackWithGeoLength(db::Features::iterator first,
                            db::Features::iterator last)
    -> std::pair<geolib3::Polyline2, double>
{
    auto track = geolib3::PointsVector{};
    auto length = 0.;
    auto areAdjacentCloseInTime = [&](auto& lhs, auto& rhs) {
        return abs(getTimestamp(lhs) - getTimestamp(rhs)) <=
               MIN_TIME_GAP_BETWEEN_RIDES;
    };
    common::forEachEqualRange(
        first,
        last,
        areAdjacentCloseInTime,
        [&track, &length](auto first, auto last) {
            auto points = makePoints(first, last);
            track.insert(track.end(), points.begin(), points.end());
            length += geolib3::geoLength(geolib3::Polyline2(points));
        });
    if (track.size() == 1) {
        // @see
        // https://stackoverflow.com/questions/62940835/how-to-make-a-postgis-linestring-with-only-one-initial-point
        track.push_back(track.front());
    }
    return {geolib3::Polyline2(std::move(track)), length};
}

}  // namespace

db::Ride makeRide(db::Features::iterator first, db::Features::iterator last)
{
    ASSERT(first != last);
    auto userId = getUserId(*first);
    auto sourceId = getSourceId(*first);
    auto clientRideId = getClientRideId(*first);
    auto minTime = getTimestamp(*first);
    auto maxTime = getTimestamp(*first);
    auto minUploadTime = getUploadedAt(*first);
    auto maxUploadTime = getUploadedAt(*first);
    auto showAuthorshipNumber = 0u;
    auto hideAuthorshipNumber = 0u;
    auto photosNumber = size_t{0};
    auto publishedPhotosNumber = size_t{0};
    auto status = db::RideStatus::Processed;
    std::for_each(first, last, [&](const auto& photo) {
        ASSERT(photo.dataset() == db::Dataset::Rides);
        requireEquals(photo, userId, sourceId, clientRideId);
        minTime = std::min(minTime, getTimestamp(photo));
        maxTime = std::max(maxTime, getTimestamp(photo));
        minUploadTime = std::min(minUploadTime, getUploadedAt(photo));
        maxUploadTime = std::max(maxUploadTime, getUploadedAt(photo));
        if (photo.processedAt().has_value()) {
            if (isShowAuthorshipEnabled(photo)) {
                ++showAuthorshipNumber;
            }
            else {
                ++hideAuthorshipNumber;
            }
        }
        ++photosNumber;
        if (photo.isPublished()) {
            ++publishedPhotosNumber;
        }
        if (!isProcessed(photo)) {
            status = db::RideStatus::Pending;
        }
    });
    auto [track, length] = makeTrackWithGeoLength(first, last);
    auto result = sql_chemistry::GatewayAccess<db::Ride>::construct()
        .setUserId(std::move(userId))
        .setSourceId(std::move(sourceId))
        .setTimes(minTime, maxTime)
        .setUploadTimes(minUploadTime, maxUploadTime)
        .setShowAuthorship(showAuthorshipNumber && !hideAuthorshipNumber)
        .setGeodeticTrack(std::move(track))
        .setDistanceInMeters(length)
        .setPhotos(photosNumber)
        .setPublishedPhotos(publishedPhotosNumber)
        .setStatus(status);
    if (clientRideId.has_value()) {
        result.setClientId(std::move(clientRideId).value());
    }
    return result;
}

namespace {

void updateRideHypotheses(sql_chemistry::Transaction& txn, const db::Ride& ride)
{
    db::RideHypothesisGateway{txn}.remove(db::table::RideHypothesis::rideId ==
                                          ride.rideId());
    db::RideHypothesisGateway{txn}.insert(makeRideHypotheses(txn, ride));
}

void update(sql_chemistry::Transaction& txn,
            db::Rides& oldRides,
            db::Rides& newRides)
{
    auto ridesPairs = std::vector<RidesPair>{};
    for (auto& oldRide : oldRides) {
        oldRide.setIsDeleted(true);
        for (auto& newRide : newRides) {
            ridesPairs.push_back({oldRide, newRide});
        }
    }
    std::sort(
        ridesPairs.begin(),
        ridesPairs.end(),
        [](const auto& lhs, const auto& rhs) { return lhs.iou() > rhs.iou(); });
    for (auto& ridesPair : ridesPairs) {
        ridesPair.assignIfPossible();
    }
    std::erase_if(newRides, isDeleted);
    for (auto& ride : oldRides) {
        if (isDeleted(ride)) {
            db::RideGateway{txn}.updatex(ride);
            updateRideHypotheses(txn, ride);
        }
    }
    for (auto& ride : oldRides) {
        if (!isDeleted(ride)) {
            db::RideGateway{txn}.updatex(ride);
            updateRideHypotheses(txn, ride);
        }
    }
    for (auto& ride : newRides) {
        db::RideGateway{txn}.insertx(ride);
        updateRideHypotheses(txn, ride);
    }
}

template <class PhotoIt>
void setShowAuthorship(bool showAuthorship,
                       PhotoIt first,
                       PhotoIt last,
                       db::Features& affectedPhotos)
{
    std::for_each(first, last, [&](const db::Feature& photo) {
        if (photo.processedAt().has_value() &&
            isShowAuthorshipEnabled(photo) != showAuthorship) {
            affectedPhotos.emplace_back(photo).setShowAuthorship(
                showAuthorship);
        }
    });
}

void updateRidesWithoutClientId(pgpool3::Pool& pool,
                                const std::string& userId,
                                const std::string& sourceId,
                                chrono::TimePoint minTime,
                                chrono::TimePoint maxTime)
{
    auto rides = db::RideGateway{*pool.masterReadOnlyTransaction()}.load(
        db::table::Ride::userId == userId &&
        db::table::Ride::sourceId == sourceId &&
        db::table::Ride::startTime <= maxTime + MIN_TIME_GAP_BETWEEN_RIDES &&
        db::table::Ride::endTime >= minTime - MIN_TIME_GAP_BETWEEN_RIDES &&
        db::table::Ride::clientId.isNull() &&
        !db::table::Ride::isDeleted.is(true));
    if (!rides.empty()) {
        minTime = std::min(
            minTime,
            std::min_element(rides.begin(), rides.end(), byStart)->startTime());
        maxTime = std::max(
            maxTime,
            std::max_element(rides.begin(), rides.end(), byEnd)->endTime());
    }
    auto photos = db::FeatureGateway{*pool.masterReadOnlyTransaction()}.load(
        db::table::Feature::dataset == db::Dataset::Rides &&
        db::table::Feature::userId == userId &&
        db::table::Feature::sourceId == sourceId &&
        db::table::Feature::date.between(minTime, maxTime) &&
        db::table::Feature::clientRideId.isNull() &&
        !db::table::Feature::deletedByUser.is(true) &&
        !db::table::Feature::gdprDeleted.is(true));
    auto newRides = db::Rides{};
    auto affectedPhotos = db::Features{};
    sortByRide(photos.begin(), photos.end());
    forEachRide(photos.begin(), photos.end(), [&](auto first, auto last) {
        const auto& newRide = newRides.emplace_back(makeRide(first, last));
        auto showAuthorship = isShowAuthorshipEnabled(newRide);
        setShowAuthorship(showAuthorship, first, last, affectedPhotos);
    });
    auto txn = pool.masterWriteableTransaction();
    update(*txn, rides, newRides);
    if (!affectedPhotos.empty()) {
        db::FeatureGateway{*txn}.update(affectedPhotos,
                                        db::UpdateFeatureTxn::Yes);
    }
    txn->commit();
}

}  // namespace

void updateRides(pgpool3::Pool& pool,
                 const std::string& userId,
                 const std::string& sourceId,
                 const std::optional<std::string>& clientRideId,
                 chrono::TimePoint minTime,
                 chrono::TimePoint maxTime)
{
    if (!clientRideId) {
        return updateRidesWithoutClientId(
            pool, userId, sourceId, minTime, maxTime);
    }

    auto ride = db::RideGateway{*pool.masterReadOnlyTransaction()}.tryLoadOne(
        db::table::Ride::clientId == *clientRideId);
    if (ride) {
        requireEquals(*ride, userId, sourceId, clientRideId);
    }
    auto photos = db::FeatureGateway{*pool.masterReadOnlyTransaction()}.load(
        db::table::Feature::dataset == db::Dataset::Rides &&
        db::table::Feature::clientRideId == *clientRideId &&
        !db::table::Feature::deletedByUser.is(true) &&
        !db::table::Feature::gdprDeleted.is(true));
    requireEquals(photos, userId, sourceId, clientRideId);
    std::sort(photos.begin(), photos.end(), common::lessFn(getTimestamp));

    auto txn = pool.masterWriteableTransaction();
    if (photos.empty()) {
        if (ride && !ride->isDeleted()) {
            ride->setIsDeleted(true);
            db::RideGateway{*txn}.updatex(*ride);
            updateRideHypotheses(*txn, *ride);
        }
    }
    else if (ride && ride->isDeleted()) {
        for (auto& photo : photos) {
            photo.setDeletedByUser(true);
        }
        db::FeatureGateway{*txn}.update(photos, db::UpdateFeatureTxn::Yes);
    }
    else {
        auto newRide = makeRide(photos.begin(), photos.end());
        auto showAuthorship = isShowAuthorshipEnabled(newRide);
        auto affectedPhotos = db::Features{};
        setShowAuthorship(
            showAuthorship, photos.begin(), photos.end(), affectedPhotos);
        db::FeatureGateway{*txn}.update(affectedPhotos,
                                        db::UpdateFeatureTxn::Yes);
        if (ride) {
            ride->copyContentFrom(newRide);
            db::RideGateway{*txn}.updatex(*ride);
            updateRideHypotheses(*txn, *ride);
        }
        else {
            db::RideGateway{*txn}.insertx(newRide);
            updateRideHypotheses(*txn, newRide);
        }
    }
    txn->commit();
}

void updateRidesByPhotoIds(pgpool3::Pool& pool, const db::TIds& photoIds)
{
    auto photos = db::FeatureGateway{*pool.masterReadOnlyTransaction()}.load(
        db::table::Feature::id.in(photoIds) &&
        db::table::Feature::dataset == db::Dataset::Rides &&
        !db::table::Feature::deletedByUser.is(true) &&
        !db::table::Feature::gdprDeleted.is(true));
    sortByRide(photos.begin(), photos.end());
    forEachRide(photos.begin(), photos.end(), [&](auto first, auto last) {
        updateRides(pool,
                    getUserId(*first),
                    getSourceId(*first),
                    getClientRideId(*first),
                    getTimestamp(*first),
                    getTimestamp(*std::prev(last)));
    });
}

}  // namespace maps::mrc::ride_inspector
