#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/metadata_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/version.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/export_gen/lib/ecstatic.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/signs_export_gen/lib/exporter.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/signs_export_gen/lib/tools.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

using namespace maps::mrc;
using namespace signs_export_gen;

namespace {

const std::string DEFAULT_TEMP_DIR = "/var/tmp/mrc_objects_export";

} // namespace

int main(int argc, const char** argv) try
{
    maps::cmdline::Parser parser(
        "This service exports detected traffic signs data to a set of fb files "
        "which are then published via ecstatic."
    );
    auto syslog = parser.string("syslog-tag")
        .help("redirect log output to syslog with given tag");
    auto configPath = parser.string("config")
        .help("path to mrc configuration");
    auto secretVersion = parser.string("secret-version")
        .help("version for mrc secrets from yav.yandex-team.ru");
    auto tempDir = parser.dir("temp-dir")
        .help("temporary directory to put export files")
        .defaultValue(DEFAULT_TEMP_DIR);
    auto dryRun = parser.flag("dry-run")
        .help("do not save changes to database nor publish the data");
    parser.parse(argc, const_cast<char**>(argv));

    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }

    auto mrcConfig =
        common::templateConfigFromCmdPath(secretVersion, configPath);
    auto poolHolder = maps::wiki::common::PoolHolder(mrcConfig.makePoolHolder(
        common::LONG_READ_DB_ID, common::LONG_READ_POOL_ID));
    maps::pgp3utils::PgAdvisoryXactMutex mutex(
        poolHolder.pool(),
        static_cast<int64_t>(common::LockId::SignsExportGenerator));
    if (!dryRun && !mutex.try_lock()) {
        INFO() << "Another process is ongoing";
        return EXIT_SUCCESS;
    }

    auto version = fb::makeVersion(maps::chrono::TimePoint::clock::now());
    INFO() << "Export version: " << version;

    auto outputPath = tempDir + "/" + ECSTATIC_DATASET_NAME;
    INFO() << "Export output path: " << outputPath;

    generateSignsExport(poolHolder.pool(), version, outputPath);

    auto ecstaticClient = common::EcstaticClient(
        common::EcstaticClient::ECSTATIC_PATH,
        mrcConfig.externals().ecstaticEnvironmentOverride());

    if (!dryRun) {
        INFO() << "Publish dataset to ecstatic";
        export_gen::publish(
            ecstaticClient, ECSTATIC_DATASET_NAME, version, outputPath);
        export_gen::waitForActivation(
            ecstaticClient, ECSTATIC_DATASET_NAME, version);

        INFO() << "Update metadata";
        auto now = maps::chrono::TimePoint::clock::now();
        updateLastRunTime(poolHolder.pool(), now);
    }

    INFO() << "Done signs export generation, version " << version;
    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << "Signs export failed: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    FATAL() << "Signs export failed: " << e.what();
    return EXIT_FAILURE;
}
