#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/unittest/env.h>

#include <maps/libs/geolib/include/point.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/signs_export_gen/lib/exporter.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/objects_reader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/traffic_sign_groups_reader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/impl/utility.h>

#include <yandex/maps/mrc/unittest/database_fixture.h>

using namespace testing;
namespace fs = std::filesystem;

namespace maps::mrc::signs_export_gen::tests {
namespace {

class Fixture : public testing::Test,
                public unittest::WithUnittestConfig<unittest::DatabaseFixture>
{
public:
    Fixture()
    {
        auto txn = pool().masterWriteableTransaction();
        txn->exec(maps::common::readFileToString(SRC_("data.sql")));
        txn->commit();
    }
};

uint64_t toMilliseconds(chrono::TimePoint timePoint)
{
    return chrono::sinceEpoch<std::chrono::milliseconds>(timePoint);
}

void checkObject(
    const fb::Object& fbObject,
    const db::eye::Object& dbObject,
    const db::eye::ObjectLocation& dbLocation)
{
    EXPECT_EQ(fbObject.id(), static_cast<uint64_t>(dbObject.id()));

    auto pos = geolib3::Point2(fbObject.posX(), fbObject.posY());
    EXPECT_TRUE(
        geolib3::test_tools::approximateEqual(
            pos,
            dbLocation.geodeticPos(),
            geolib3::EPS));

    auto dbHeading = eye::decomposeRotation(dbLocation.rotation()).heading.value();
    EXPECT_EQ(fbObject.heading(), static_cast<unsigned short>(dbHeading));

    EXPECT_EQ(fb::decode(fbObject.type()), dbObject.type());

    if (dbObject.disappearedAt().has_value()) {
        EXPECT_EQ(fbObject.disappearedAt(), toMilliseconds(dbObject.disappearedAt().value()));
    } else {
        EXPECT_EQ(fbObject.disappearedAt(), 0ul);
    }
}

void checkObjectsIndex(const fb::ObjectsReader& reader, pgpool3::Pool& pool)
{
    EXPECT_EQ(reader.objectsNumber(), 4u);

    auto txn = pool.slaveTransaction();

    // present object ids
    for (int objectId : {1, 2, 3, 4}) {
        auto dbObject = db::eye::ObjectGateway{*txn}.loadById(objectId);
        auto dbObjectLocation = db::eye::ObjectLocationGateway{*txn}.loadById(objectId);
        auto written = reader.objectById(objectId);
        EXPECT_TRUE(written);
        checkObject(*written, dbObject, dbObjectLocation);
    }

    // missing object ids: 5 - deleted, 11, 12 - additional tables, 700 - traffic light
    for (int objectId : {5, 11, 12, 700}) {
        EXPECT_EQ(reader.objectById(objectId), nullptr);
    }
}

auto findFrameByFeatureId(const db::eye::Frames& frames, db::TId featureId)
{
    return std::find_if(frames.begin(), frames.end(), [&](const auto& frame) {
        auto urlContext = frame.urlContext();
        return urlContext.source() == db::eye::FrameSource::Mrc
            && urlContext.mrc().featureId == featureId;
    });
}

auto findFrameByPanoramaOid(const db::eye::Frames& frames, db::PanoramaOID oid)
{
    return std::find_if(frames.begin(), frames.end(), [&](const auto& frame) {
        auto urlContext = frame.urlContext();
        return urlContext.source() == db::eye::FrameSource::Panorama
            && urlContext.panorama().oid == oid;
    });
}

void checkFrames(const auto* fbFrames, const auto& dbFrames)
{
    if (dbFrames.empty()) {
        EXPECT_FALSE(fbFrames);
    } else {
        ASSERT_TRUE(fbFrames);
        ASSERT_EQ(fbFrames->size(), dbFrames.size());
        for (size_t i = 0; i < fbFrames->size(); ++i) {
            auto fbFrame = fbFrames->Get(i);
            if (fbFrame->source_type() == fb::FrameSource_SourceFeature) {
                auto fbFeature = static_cast<const fb::SourceFeature*>(fbFrame->source());
                auto itr = findFrameByFeatureId(dbFrames, fbFeature->featureId());
                EXPECT_NE(itr, dbFrames.end());
            } else if (fbFrame->source_type() == fb::FrameSource_SourcePanorama) {
                auto fbPanorama = static_cast<const fb::SourcePanorama*>(fbFrame->source());
                auto itr = findFrameByPanoramaOid(dbFrames, fbPanorama->oid()->str());
                EXPECT_NE(itr, dbFrames.end());

                auto dbPanoAttrs = itr->urlContext().panorama();
                EXPECT_EQ(fbPanorama->heading(), dbPanoAttrs.heading.value());
                EXPECT_EQ(fbPanorama->tilt(), dbPanoAttrs.tilt.value());
                EXPECT_EQ(fbPanorama->horizontalFov(), dbPanoAttrs.horizontalFOV.value());
                EXPECT_EQ(fbPanorama->width(), dbPanoAttrs.size.width);
                EXPECT_EQ(fbPanorama->height(), dbPanoAttrs.size.height);
            }
        }
    }
}

void checkInformationTables(const auto* fbInfoTables, const auto& dbObjects)
{
    if (dbObjects.empty()) {
        EXPECT_FALSE(fbInfoTables);
    } else {
        ASSERT_TRUE(fbInfoTables);
        EXPECT_EQ(fbInfoTables->size(), dbObjects.size());

        std::vector<traffic_signs::TrafficSign> fbInfoTableTypes;
        for (size_t i = 0; i < fbInfoTables->size(); ++i) {
            fbInfoTableTypes.push_back(fb::decode(
                static_cast<fb::TrafficSignType>(fbInfoTables->Get(i))));
        }

        std::vector<traffic_signs::TrafficSign> dbInfoTableTypes;
        for (const db::eye::Object& object : dbObjects) {
            dbInfoTableTypes.push_back(
                object.attrs<db::eye::SignAttrs>().type);
        }

        EXPECT_THAT(
            fbInfoTableTypes,
            ::testing::UnorderedElementsAreArray(dbInfoTableTypes));
    }
}

void checkObject(
    const fb::ObjectsReader& reader,
    db::TId objectId,
    const db::TIds& visibleOnFrameIds,
    const db::TIds& missingOnFrameIds,
    const db::TIds& infoTablesObjectIds,
    pgpool3::Pool& pool)
{
    auto fbObject = reader.objectById(objectId);

    auto txn = pool.slaveTransaction();
    auto frameGtw = db::eye::FrameGateway{*txn};
    auto objectGtw = db::eye::ObjectGateway{*txn};

    db::eye::Object dbObject = objectGtw.loadById(objectId);
    auto dbFramesVisible = frameGtw.loadByIds(visibleOnFrameIds);
    auto dbFramesMissing = frameGtw.loadByIds(missingOnFrameIds);
    auto dbInfoTables = objectGtw.loadByIds(infoTablesObjectIds);

    // Check attrs
    ASSERT_TRUE(fbObject->attrs());
    EXPECT_EQ(fbObject->attrs_type(), fb::ObjectAttrs_TrafficSignAttrs);
    auto attrs = static_cast<const fb::TrafficSignAttrs*>(fbObject->attrs());
    EXPECT_EQ(fb::decode(attrs->trafficSignType()),
              dbObject.attrs<db::eye::SignAttrs>().type);

    // Check visible on frames
    checkFrames(fbObject->visibleOnFrames(), dbFramesVisible);

    // Check missing on frames
    checkFrames(fbObject->missingOnFrames(), dbFramesMissing);

    // Check information tables
    checkInformationTables(fbObject->informationTables(), dbInfoTables);
}

void checkSignGroups(
    const fb::ObjectsReader& objectsReader,
    const fb::TrafficSignGroupsReader& groupsReader)
{
    EXPECT_EQ(groupsReader.groupsNumber(), 1u);
    auto group = groupsReader.group(0);
    ASSERT_TRUE(group);
    EXPECT_EQ(group->orientation(), fb::GroupOrientation::GroupOrientation_Vertical);
    ASSERT_EQ(group->objectIds()->size(), 2u);
    EXPECT_EQ(group->objectIds()->Get(0), 2);
    EXPECT_EQ(group->objectIds()->Get(1), 3);

    for (auto objectId : {2, 3}) {
        auto object = objectsReader.objectById(objectId);
        ASSERT_TRUE(object);
        EXPECT_EQ(object->signsGroupId(), 1u);
    }

    // Other objects are not in any group
    for (auto objectId : {1, 4}) {
        auto object = objectsReader.objectById(objectId);
        ASSERT_TRUE(object);
        EXPECT_EQ(object->signsGroupId(), 0u);
    }
}

void checkRtree(const fb::ObjectsReader& reader)
{
    auto box = geolib3::BoundingBox(
        geolib3::Point2{44.008681, 56.317259},
        geolib3::Point2{44.020970, 56.325399}
    );
    auto objects = reader.objectsInBbox(box);
    ASSERT_EQ(objects.size(), 1u);
    EXPECT_TRUE(objects.front());
    EXPECT_EQ(objects.front()->id(), 4u);
}

} // namespace

TEST_F(Fixture, test_signs_export_gen)
{
    auto frameIds = db::eye::FrameGateway{*pool().slaveTransaction()}.loadIds(
        db::eye::table::Frame::deleted.is(false));
    EXPECT_EQ(frameIds.size(), 7u);

    const std::string exportDir = GetWorkPath() + "/output_fb_files";
    fs::create_directories(exportDir);

    const std::string VERSION = "test-version";

    generateSignsExport(pool(), VERSION, exportDir);

    auto reader = fb::ObjectsReader{exportDir};
    checkObjectsIndex(reader, pool());

    //                  id   visible   missing  info tables
    //                  -----------------------------------
    checkObject(reader, 1,   {1,2,3},  {},      {11,12},   pool());
    checkObject(reader, 2,   {1,4},    {},      {},        pool());
    checkObject(reader, 3,   {1,4},    {},      {},        pool());
    checkObject(reader, 4,   {1},      {5},     {},        pool());

    auto groupsReader = fb::TrafficSignGroupsReader{exportDir};
    checkSignGroups(reader, groupsReader);

    checkRtree(reader);
}

} // maps::mrc::signs_export_gen::tests
