#include "generate_tasks.h"
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/mongo/include/init.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/tools/grinder/worker/include/api.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/wiki_config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/revision_loader.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

namespace maps::mrc::gen_targets {

void generateTasksHandler(pgpool3::Pool& pool,
                          object::Loader& objectLoader,
                          const grinder::worker::Task& task)
{
    db::TId tasksGroupId = task.args()["tasks_group_id"].as<db::TId>();
    INFO() << "Generating tasks for tasks group " << tasksGroupId;
    StaticGraphLoader graphLoader;
    generateTasks(pool, objectLoader, graphLoader, tasksGroupId);
}

} // namespace maps::mrc::gen_targets

int main(int argc, char* argv[]) try {
    using namespace maps;
    maps::cmdline::Parser parser;
    auto configPath
        = parser.string("config").help("path to worker configuration");
    auto wikiConfigDirPath = parser.string("wiki-config-dir")
            .defaultValue(maps::mrc::common::WIKI_TEMPLATE_CONFIG_DIR)
            .help("path to services config template for wikimap");
    auto wikiConfigPath =
        parser.string("wiki-config")
            .help("path to services config for wikimap");
    auto syslogTag
        = parser.string("syslog-tag")
              .help("redirect log output to syslog with given tag");
    auto grinderConfigPath = parser.string("grinder-config")
                                 .help("path to grinder configuration file");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    maps::mongo::init();

    grinder::worker::Options workerOpts;

    auto config = maps::mrc::common::templateConfigFromCmdPath(configPath);
    auto pgpoolHolder =
        config.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                              maps::mrc::common::LONG_READ_POOL_ID);

    auto wikiConfig = wikiConfigPath.defined()
        ? maps::wiki::common::ExtendedXmlDoc(wikiConfigPath)
        : maps::mrc::common::makeWikiConfig(wikiConfigDirPath);
    maps::wiki::common::PoolHolder wikiPoolHolder{wikiConfig, "long-read", "long-read"};

    if (grinderConfigPath.defined()) {
        workerOpts.setConfigLocation(grinderConfigPath);
    }

    workerOpts.on("generate_tasks",
        [&](const grinder::worker::Task& task) {
            auto objectLoader = mrc::object::makeRevisionLoader(wikiPoolHolder.pool().slaveTransaction());
            maps::mrc::gen_targets::generateTasksHandler(pgpoolHolder.pool(), *objectLoader, task);
        }
    ).setConcurrencyLevel(5);

    grinder::worker::run(workerOpts);

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
