#include "generate_tasks.h"

#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/tasks_group_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/write.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/mock_loader.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/data_model.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/districts.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/static_graph_loader.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/tasks_generator.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <yandex/maps/jams/static_graph2/types.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

#include <algorithm>
#include <future>
#include <vector>

namespace maps::mrc::gen_targets::tests {
namespace {

struct Playground {
    static Playground& instance() {
        static Playground playground;
        return playground;
    }

    unittest::WithUnittestConfig<unittest::DatabaseFixture> databaseFixture;
};

static const auto TEST_GRAPH_PATH = BinaryPath("maps/data/test/graph4");

using CoverageMap = std::unordered_map<db::CameraDeviation, fb::TEdgeCoverage>;

using EdgeMap = std::unordered_map<EdgeId, CoverageMap>;

EdgeMap makeEdgeMap(const fb::GraphReader& reader)
{
    EdgeMap result;
    for (size_t i = 0; i < reader.edgesNumber(); ++i) {
        auto edge = reader.edge(i);
        for (auto& coverage : edge.coverages) {
            result[edge.id][coverage.cameraDeviation] = coverage;
        }
    }
    return result;
}

fb::TGraph makeGraph(const std::string& version, const EdgeMap& edgeMap)
{
    fb::TGraph result{.version = version};
    for (auto& [edgeId, coverageMap] : edgeMap) {
        auto& edge = result.edges.emplace_back();
        edge.id = edgeId;
        for (auto& [cameraDeviation, coverage] : coverageMap) {
            edge.coverages.push_back(coverage);
        }
    }
    return result;
}

class Fixture : public testing::Test
{
public:
    Fixture()
    {
        Playground::instance().databaseFixture.postgres().truncateTables();
        fb::TGraph graph{
            .version = static_cast<std::string>(
                road_graph::Graph(TEST_GRAPH_PATH + "/road_graph.fb")
                    .version())};
        fb::writeToFile(graph, TEST_GRAPH_PATH + "/graph_coverage.fb");
        graphLoader_ = std::make_shared<StaticGraphLoader>(TEST_GRAPH_PATH);
    }

    auto graphLoader() {
        return graphLoader_;
    }

    pgpool3::Pool& pool() {
        return Playground::instance().databaseFixture.pool();
    }

    void createGraphCoverageForEdgesInRegion(
        const geolib3::Polygon2& geodeticGeom,
        float coverageFraction,
        chrono::TimePoint actualizationDate,
        db::CameraDeviation cameraDeviation = db::CameraDeviation::Front)
    {
        auto edgeMap = makeEdgeMap(graphLoader_->coverage());
        auto roadNetwork = graphLoader_->loadGraph(geodeticGeom.boundingBox(),
                                                   7,     // allowedUturnsFc
                                                   false  // allowTollRoads
        );
        auto edgeIds = selectTargetEdges(
            roadNetwork, geodeticGeom, [](const Edge&) { return true; });
        for (auto edgeId : edgeIds) {
            edgeMap[edgeId][cameraDeviation] =
                fb::TEdgeCoverage{.coverageFraction = coverageFraction,
                                  .actualizationDate = actualizationDate,
                                  .cameraDeviation = cameraDeviation};
        }
        auto graph = makeGraph(
            static_cast<std::string>(graphLoader_->coverage().version()),
            edgeMap);
        fb::writeToFile(graph, TEST_GRAPH_PATH + "/graph_coverage.fb");
        graphLoader_ = std::make_shared<StaticGraphLoader>(TEST_GRAPH_PATH);
    }

    object::MockLoader& objectLoader() { return objectLoader_; }

private:
    std::shared_ptr<StaticGraphLoader> graphLoader_;
    object::MockLoader objectLoader_;
};

using geolib3::Polygon2;
using geolib3::MultiPolygon2;
using geolib3::PointsVector;

const Polygon2 EXCLUSION_REGION_GEODETIC_GEOM(
        PointsVector{
            {37.5826, 55.7362},
            {37.5892, 55.7318},
            {37.5900, 55.7323},
            {37.5836, 55.7368}
        }
    );

const Polygon2 TARGET_REGION1(
        PointsVector{
            {37.5542, 55.7226},
            {37.5707, 55.7180},
            {37.5979, 55.7348},
            {37.5878, 55.7402},
            {37.5594, 55.7281}
        }
    );

const Polygon2 TARGET_REGION2(
        PointsVector{
            {37.5891, 55.7410},
            {37.6000, 55.7353},
            {37.6086, 55.7390},
            {37.6090, 55.7429},
            {37.6004, 55.7459}
        }
    );

const Polygon2 REGION_OF_DEADENDS(
    PointsVector{
        {37.543747, 55.586108},
        {37.558102, 55.584236},
        {37.550291, 55.582547},
        {37.544176, 55.584965}
    }
);


db::ugc::TasksGroup createExampleTasksGroup(
    db::ugc::UseRouting useRouting,
    const std::vector<geolib3::Polygon2>& polygons,
    boost::optional<chrono::TimePoint> actualizedBefore = boost::none,
    const std::vector<db::CameraDeviation>& cameraDeviations = {},
    std::optional<float> minEdgeCoverageRatio = {},
    db::ugc::TasksGroupStatus status = db::ugc::TasksGroupStatus::Generating)
{
    db::ugc::TasksGroup group{
        status,
        "Moscow 2018.02",
        geolib3::convertGeodeticToMercator(MultiPolygon2(polygons)),
        useRouting,
        {1, 2, 3, 4, 5, 6, 7},
        db::ugc::UseToll::No,
        10000 // recommended length
    };

    if (!cameraDeviations.empty()) {
        group.setCameraDeviations(cameraDeviations);
    }

    if (actualizedBefore) {
        group.setActualizedBefore(*actualizedBefore);
    }

    if (minEdgeCoverageRatio.has_value()) {
        group.setMinEdgeCoverageRatio(minEdgeCoverageRatio.value());
    }
    return group;
}

void saveTasksGroup(
    maps::pgpool3::TransactionHandle&& txn, db::ugc::TasksGroup& tasksGroup)
{
    db::ugc::TasksGroupGateway gtw(*txn);
    gtw.insert(tasksGroup);
    txn->commit();
}

db::ugc::TasksGroup createAndSaveExampleTasksGroup(
    maps::pgpool3::TransactionHandle&& txn,
    db::ugc::UseRouting useRouting,
    const std::vector<geolib3::Polygon2>& polygons,
    boost::optional<chrono::TimePoint> actualizedBefore = boost::none,
    const std::vector<db::CameraDeviation>& cameraDeviations = {},
    std::optional<float> minEdgeCoverageRatio = {},
    db::ugc::TasksGroupStatus status = db::ugc::TasksGroupStatus::Generating)
{
    auto tasksGroup = createExampleTasksGroup(
        useRouting,
        polygons,
        actualizedBefore,
        cameraDeviations,
        minEdgeCoverageRatio,
        status);
    saveTasksGroup(std::move(txn), tasksGroup);
    return tasksGroup;
}

size_t countTaskTargetsInsideRegion(pqxx::transaction_base& txn,
                                    db::TId tasksGroupId,
                                    const geolib3::Polygon2& geodeticGeom)
{
    auto targets = db::ugc::TargetGateway(txn).load(
        db::ugc::table::Target::geom.intersects<geolib3::SpatialReference::Epsg4326>(geodeticGeom) &&
        db::ugc::table::Target::taskId == db::ugc::table::Task::id &&
        db::ugc::table::Task::tasksGroupId == tasksGroupId
    );

    return std::count_if(targets.begin(), targets.end(),
        [&](const db::ugc::Target& target) {
            return geolib3::spatialRelation(geodeticGeom, target.geodeticGeom(),
                                            geolib3::SpatialRelation::Intersects);
        });
}

void checkTask(pqxx::transaction_base& txn,
               const db::ugc::TasksGroup& tasksGroup,
               const db::ugc::Task& task)
{
    EXPECT_EQ(task.status(), db::ugc::TaskStatus::Draft);
    EXPECT_GT(task.durationInSeconds().count(), 0);
    EXPECT_GT(task.distanceInMeters(), 0.);
    EXPECT_GT(task.geodeticHull().area(), 0.);

    auto names = db::ugc::TaskNameGateway(txn).load(
        db::ugc::table::TaskName::taskId == task.id()
    );
    EXPECT_GT(names.size(), 0u);
    EXPECT_EQ(names[0].value().find(tasksGroup.name()), 0u);

    auto targets = db::ugc::TargetGateway(txn).load(
        db::ugc::table::Target::taskId == task.id()
    );
    EXPECT_GT(targets.size(), 0u);

    if (tasksGroup.useRouting() == db::ugc::UseRouting::Yes){
        EXPECT_TRUE(targets[0].forwardPos());
    } else {
        EXPECT_FALSE(targets[0].forwardPos());
    }

    EXPECT_EQ(task.cameraDeviations(), tasksGroup.cameraDeviations());
}

} // namespace


TEST_F(Fixture, test_generate_routing_tasks)
{
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::Yes,
        {TARGET_REGION1});

    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

    auto txn = pool().masterReadOnlyTransaction();
    db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
    EXPECT_TRUE(tasksGroup.totalLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.uniqueLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.graphCoverageRatio().value() > 0.);
    EXPECT_EQ(
        59u,
        countTaskTargetsInsideRegion(
            *txn, tasksGroup.id(), EXCLUSION_REGION_GEODETIC_GEOM));
    auto tasks = db::ugc::TaskGateway(*txn).load(
        db::ugc::table::Task::tasksGroupId == tasksGroup.id());
    EXPECT_EQ(7u, tasks.size());
    checkTask(*txn, tasksGroup, tasks[0]);
}

TEST_F(Fixture, test_residential_regions_considered)
{
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::Yes,
        {TARGET_REGION1});

    objectLoader().add(
        object::MrcRegions{
            object::MrcRegion(
                object::RevisionID{201, 1},
                geolib3::convertGeodeticToMercator(EXCLUSION_REGION_GEODETIC_GEOM)
            ).type(object::MrcRegion::Type::Residential)
        }
    );
    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

    auto txn = pool().masterReadOnlyTransaction();
    db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
    EXPECT_TRUE(tasksGroup.totalLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.uniqueLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.graphCoverageRatio().value() > 0.);
    EXPECT_LT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), EXCLUSION_REGION_GEODETIC_GEOM), 45u);
}

TEST_F(Fixture, test_residential_regions_ignored)
{
    auto tasksGroup =
        createExampleTasksGroup(db::ugc::UseRouting::Yes, {TARGET_REGION1});
    tasksGroup.setIgnorePrivateArea(true);
    saveTasksGroup(pool().masterWriteableTransaction(),tasksGroup);

    objectLoader().add(
        object::MrcRegions{
            object::MrcRegion(
                object::RevisionID{201, 1},
                geolib3::convertGeodeticToMercator(EXCLUSION_REGION_GEODETIC_GEOM)
            ).type(object::MrcRegion::Type::Residential)
        }
    );
    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

    auto txn = pool().masterReadOnlyTransaction();
    db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
    EXPECT_TRUE(tasksGroup.totalLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.uniqueLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.graphCoverageRatio().value() > 0.);
    EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), EXCLUSION_REGION_GEODETIC_GEOM), 45u);
}

TEST_F(Fixture, test_restricted_regions_considered)
{
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::Yes,
        {TARGET_REGION1});

    objectLoader().add(
        object::MrcRegions{
            object::MrcRegion(
                object::RevisionID{201, 1},
                geolib3::convertGeodeticToMercator(EXCLUSION_REGION_GEODETIC_GEOM)
            ).type(object::MrcRegion::Type::Restricted)
        }
    );
    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

    auto txn = pool().masterReadOnlyTransaction();
    db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
    EXPECT_TRUE(tasksGroup.totalLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.uniqueLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.graphCoverageRatio().value() > 0.);
    EXPECT_LT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), EXCLUSION_REGION_GEODETIC_GEOM), 5u);
}

TEST_F(Fixture, test_generate_tasks_without_routing_over_a_single_region)
{
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::No,
        {TARGET_REGION1});

    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

    auto txn = pool().masterReadOnlyTransaction();
    db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
    EXPECT_TRUE(tasksGroup.totalLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.uniqueLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.graphCoverageRatio().value() > 0.);
    EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), EXCLUSION_REGION_GEODETIC_GEOM), 0u);

    auto tasks = db::ugc::TaskGateway(*txn).load(db::ugc::table::Task::tasksGroupId == tasksGroup.id());
    EXPECT_EQ(tasks.size(), 1u);
    checkTask(*txn, tasksGroup, tasks[0]);
}

TEST_F(Fixture, test_generate_tasks_without_routing_over_two_regions)
{
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::No,
        {TARGET_REGION1, TARGET_REGION2});

    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

    auto txn = pool().masterReadOnlyTransaction();
    db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
    EXPECT_TRUE(tasksGroup.totalLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.uniqueLengthMeters().value() > 0u);
    EXPECT_TRUE(tasksGroup.graphCoverageRatio().value() > 0.);
    EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), EXCLUSION_REGION_GEODETIC_GEOM), 0u);

    auto tasks = db::ugc::TaskGateway(*txn).load(db::ugc::table::Task::tasksGroupId == tasksGroup.id());
    EXPECT_EQ(tasks.size(), 2u);
}

TEST_F(Fixture, test_tasks_group_actualized_before_considered)
{
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::No,
        {TARGET_REGION1, TARGET_REGION2},
        chrono::parseSqlDateTime("2018-09-01 00:00:00"),
        {},
        0.5);

    createGraphCoverageForEdgesInRegion(TARGET_REGION1,
                                        0.6 /* coverageFraction */,
                                        chrono::parseSqlDateTime("2018-09-02 00:00:00") /* actualizationDate */);

    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());
    auto txn = pool().masterReadOnlyTransaction();
    EXPECT_EQ(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), TARGET_REGION1), 0u);
}

TEST_F(Fixture, test_exclude_deadends_considered)
{
    {
        auto tasksGroup = createAndSaveExampleTasksGroup(
            pool().masterWriteableTransaction(),
            db::ugc::UseRouting::No,
            {REGION_OF_DEADENDS});

        generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

        auto txn = pool().masterReadOnlyTransaction();
        db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
        EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
        EXPECT_TRUE(tasksGroup.totalLengthMeters().value() > 0u);
        EXPECT_TRUE(tasksGroup.uniqueLengthMeters().value() > 0u);
        EXPECT_TRUE(tasksGroup.graphCoverageRatio().value() > 0.);
        EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), REGION_OF_DEADENDS), 0u);
    }

    {
        auto tasksGroup = createExampleTasksGroup(
            db::ugc::UseRouting::No, {TARGET_REGION1});
        tasksGroup.setExcludeDeadends(true);
        saveTasksGroup(pool().masterWriteableTransaction(), tasksGroup);

        generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

        auto txn = pool().masterReadOnlyTransaction();
        db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
        EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
        EXPECT_TRUE(tasksGroup.totalLengthMeters().value() > 0u);
        EXPECT_TRUE(tasksGroup.uniqueLengthMeters().value() > 0u);
        EXPECT_TRUE(tasksGroup.graphCoverageRatio().value() > 0.);
        EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), TARGET_REGION1), 0u);
    }

    {
        auto tasksGroup = createExampleTasksGroup(
            db::ugc::UseRouting::No, {REGION_OF_DEADENDS});
        tasksGroup.setExcludeDeadends(true);
        saveTasksGroup(pool().masterWriteableTransaction(), tasksGroup);

        {
            auto txn = pool().masterWriteableTransaction();
            db::ugc::TasksGroupGateway(*txn).update(tasksGroup);
            txn->commit();
        }
        generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());

        auto txn = pool().masterReadOnlyTransaction();
        db::ugc::TasksGroupGateway(*txn).reload(tasksGroup);
        EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
        EXPECT_EQ(tasksGroup.totalLengthMeters().value(),  0u);
        EXPECT_EQ(tasksGroup.uniqueLengthMeters().value(), 0u);
        EXPECT_EQ(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), REGION_OF_DEADENDS), 0u);
    }
}

TEST_F(Fixture, test_tasks_group_actualized_before_coverage_obsolete)
{
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::No,
        {TARGET_REGION1, TARGET_REGION2},
        chrono::parseSqlDateTime("2018-09-01 00:00:00"));

    createGraphCoverageForEdgesInRegion(TARGET_REGION1,
                                        1.0 /* coverageFraction */,
                                        chrono::parseSqlDateTime("2018-08-31 00:00:00") /* actualizationDate */);
    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());
    auto txn = pool().masterReadOnlyTransaction();
    EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), TARGET_REGION1), 0u);
}

TEST_F(Fixture, test_tasks_group_actualized_before_coverage_insufficient)
{

    auto tasksGroup = createAndSaveExampleTasksGroup(pool().masterWriteableTransaction(),
                                              db::ugc::UseRouting::No,
                                              {TARGET_REGION1, TARGET_REGION2},
                                              chrono::parseSqlDateTime("2018-09-01 00:00:00"));
    createGraphCoverageForEdgesInRegion(TARGET_REGION1,
                                        0.5 /* coverageFraction */,
                                        chrono::parseSqlDateTime("2018-09-02 00:00:00") /* actualizationDate */);

    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());
    auto txn = pool().masterReadOnlyTransaction();
    EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), TARGET_REGION1), 0u);
}


TEST_F(Fixture, test_tasks_group_with_two_camera_deviations_coverage_ok)
{

    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::No,
        {TARGET_REGION1, TARGET_REGION2},
        chrono::parseSqlDateTime("2018-09-01 00:00:00"),
        {db::CameraDeviation::Front, db::CameraDeviation::Right});

    createGraphCoverageForEdgesInRegion(
        TARGET_REGION1,
        0.95 /* coverageFraction */,
        chrono::parseSqlDateTime("2018-09-02 00:00:00") /* actualizationDate */,
        db::CameraDeviation::Front);

    createGraphCoverageForEdgesInRegion(
        TARGET_REGION1,
        0.95 /* coverageFraction */,
        chrono::parseSqlDateTime("2018-09-02 00:00:00") /* actualizationDate */,
        db::CameraDeviation::Right);

    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());
    auto txn = pool().masterReadOnlyTransaction();
    EXPECT_EQ(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), TARGET_REGION1), 0u);

    auto tasks = db::ugc::TaskGateway(*txn).load(
        db::ugc::table::Task::tasksGroupId == tasksGroup.id());
    EXPECT_GT(tasks.size(), 0u);
    checkTask(*txn, tasksGroup, tasks[0]);
}

TEST_F(Fixture, test_tasks_group_camera_deviation_coverage_insufficient)
{

    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::No,
        {TARGET_REGION1, TARGET_REGION2},
        chrono::parseSqlDateTime("2018-09-01 00:00:00"),
        {db::CameraDeviation::Front, db::CameraDeviation::Right});

    createGraphCoverageForEdgesInRegion(
        TARGET_REGION1,
        0.95 /* sufficient coverageFraction */,
        chrono::parseSqlDateTime("2018-09-02 00:00:00") /* actualizationDate */,
        db::CameraDeviation::Front);

    createGraphCoverageForEdgesInRegion(
        TARGET_REGION1,
        0.65 /* insufficient coverageFraction */,
        chrono::parseSqlDateTime("2018-09-02 00:00:00") /* actualizationDate */,
        db::CameraDeviation::Right);

    createGraphCoverageForEdgesInRegion(
        TARGET_REGION2,
        1.0 /* coverageFraction */,
        chrono::parseSqlDateTime("2018-09-02 00:00:00") /* actualizationDate */,
        db::CameraDeviation::Front);

    createGraphCoverageForEdgesInRegion(
        TARGET_REGION2,
        1.0 /* coverageFraction */,
        chrono::parseSqlDateTime("2018-01-01 00:00:00") /* actualizationDate */,
        db::CameraDeviation::Right);

    generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());
    auto txn = pool().masterReadOnlyTransaction();
    EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), TARGET_REGION1), 0u);
    EXPECT_GT(countTaskTargetsInsideRegion(*txn, tasksGroup.id(), TARGET_REGION2), 0u);
}

TEST_F(Fixture, test_concurrent_generate_tasks)
{
    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::Yes,
        {TARGET_REGION1});

    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Generating);
    auto future1 = std::async(std::launch::async, [&] {
        return generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());
    });
    auto future2 = std::async(std::launch::async, [&] {
        return generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());
    });
    EXPECT_FALSE(future1.get() && future2.get());
    auto txn = pool().masterReadOnlyTransaction();
    db::ugc::TasksGroupGateway{*txn}.reload(tasksGroup);
    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
    EXPECT_EQ(
        59u,
        countTaskTargetsInsideRegion(
            *txn, tasksGroup.id(), EXCLUSION_REGION_GEODETIC_GEOM));
    auto tasks = db::ugc::TaskGateway(*txn).load(
        db::ugc::table::Task::tasksGroupId == tasksGroup.id());
    EXPECT_EQ(7u, tasks.size());
}

TEST_F(Fixture, test_late_status)
{
    /// @see https://st.yandex-team.ru/MAPSMRC-2461

    auto tasksGroup = createAndSaveExampleTasksGroup(
        pool().masterWriteableTransaction(),
        db::ugc::UseRouting::Yes,
        {TARGET_REGION1},
        boost::none, // actualizedBefore
        {},          // cameraDeviations
        {},          // minEdgeCoverageRatio
        db::ugc::TasksGroupStatus::Draft);

    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Draft);

    auto future = std::async(std::launch::async, [&] {
        return generateTasks(pool(), objectLoader(), *graphLoader(), tasksGroup.id());
    });
    EXPECT_FALSE(future.wait_for(std::chrono::seconds(1)) ==
                 std::future_status::ready);

    {  // change status later
        auto txn = pool().masterWriteableTransaction();
        db::ugc::TasksGroupGateway gtw(*txn);
        gtw.update(tasksGroup.setStatus(db::ugc::TasksGroupStatus::Generating));
        txn->commit();
    }

    EXPECT_TRUE(future.get());
    db::ugc::TasksGroupGateway{*pool().masterReadOnlyTransaction()}.reload(
        tasksGroup);
    EXPECT_EQ(tasksGroup.status(), db::ugc::TasksGroupStatus::Open);
}

} // namespace maps::mrc::gen_targets::tests
