#include "data_model.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/geolib/include/prepared_polygon.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <unordered_set>
#include <queue>

namespace maps::mrc::gen_targets {

namespace {

const Seconds rightPenalty = 10;
const Seconds leftPenalty = 20;
const Seconds uTurnPenalty = 30;

}

RoadNetworkData::RoadNetworkData(Edges&& edges)
    : edges_(std::move(edges))
{
    for (auto& edge : edges_) {
        calculateOutputPenalties(edge.second);
    }
    INFO() << "created road network with "
           << edges_.size() << " edges";
}

Meters RoadNetworkData::selectTargetsAsPolygon(
    const geolib3::Polygon2& polygon,
    int maxRoadFC,
    const std::unordered_set<EdgeId>& prohibitedEdges)
{
    std::vector<EdgeId> targetEdges = getEdgesWithinPolygon(polygon,
                                                            0,
                                                            maxRoadFC,
                                                            prohibitedEdges);
    Meters lengthSum = 0;
    for (EdgeId edgeId : targetEdges) {
        Edge& edge = edges_[edgeId];
        edge.isTarget = true;
        lengthSum += edge.length;
    }

    return lengthSum;
}

std::vector<EdgeId> RoadNetworkData::getEdgesWithinPolygon(
    const geolib3::Polygon2& polygon,
    int minRoadFC,
    int maxRoadFC,
    const std::unordered_set<EdgeId>& prohibitedEdges) const
{
    INFO() << "Selecting edges within polygon...";
    auto bbox = polygon.boundingBox();
    geolib3::PreparedPolygon2 preparedPolygon(polygon);

    std::vector<EdgeId> targetEdges;

    for (auto& it : edges_) {
        const Edge& edge = it.second;
        if (edge.fc <= maxRoadFC
            && edge.fc >= minRoadFC
            && !edge.isUTurn // U turns connectors are not interesting for driving
            && !prohibitedEdges.count(edge.id)
            && polylineIsMostlyInsidePolygon(edge.geom, preparedPolygon, bbox)
            ) {
            targetEdges.push_back(edge.id);
        }
    }

    INFO() << "edges within polygon were selected";
    return targetEdges;
}



boost::iterator_range<Edges::const_iterator> RoadNetworkData::edges() const
{
    return boost::make_iterator_range(edges_.cbegin(), edges_.cend());
}

const Edge& RoadNetworkData::edge(EdgeId id) const
{
    auto it = edges_.find(id);
    REQUIRE(it != edges_.end(), "Edge " << id << " not found");

    return it->second;
}

std::pair<EdgeCRefVec, std::vector<Seconds>>
RoadNetworkData::getOutEdgesAndPenalties(EdgeId edgeId) const
{
    std::pair<EdgeCRefVec, std::vector<Seconds>> result;
    auto& edge = edges_.at(edgeId);
    for(size_t i = 0; i < edge.outEdges.size(); i++) {
        result.first.emplace_back(std::cref(edges_.at(edge.outEdges[i])));
        result.second.push_back(edge.outEdgesPenalties[i]);
    }
    return result;
}

Seconds RoadNetworkData::getManeuverPenalty(EdgeId from, EdgeId to) const
{
    auto& fromEdge = edges_.at(from);
    for (size_t i = 0; i < fromEdge.outEdges.size(); i++) {
        if (fromEdge.outEdges[i] == to) {
            return fromEdge.outEdgesPenalties[i];
        }
    }
    throw maps::Exception("edges are not connected");
}

std::unordered_set<EdgeId> RoadNetworkData::getSetOfEdgeIds() const
{
    std::unordered_set<EdgeId> allEdgeIds;
    allEdgeIds.reserve(edges_.size());
    for (const auto& edgeIt : edges_) {
        allEdgeIds.insert(edgeIt.first);
    }
    return allEdgeIds;
}


void RoadNetworkData::calculateOutputPenalties(Edge& edge) {
    bool straightWayIsAvailable = false;
    bool rightTurnAvailable = false;
    for (auto outEdgeId : edge.outEdges) {
        geolib3::Direction2 curEdgeDirection = edge.outgoingDirection();
        geolib3::Direction2 outEdgeDirection = edges_[outEdgeId].incomingDirection();
        if (turnIsStraight(curEdgeDirection, outEdgeDirection)) {
            straightWayIsAvailable = true;
        }
        if (turnIsRight(curEdgeDirection, outEdgeDirection)) {
            rightTurnAvailable = true;
        }
    }

    edge.outEdgesPenalties.resize(edge.outEdges.size());
    for (size_t i = 0; i < edge.outEdges.size(); i++) {
        auto outEdge = edges_[edge.outEdges[i]];
        Seconds penalty = 0;
        geolib3::Direction2 fromDirection = edge.outgoingDirection();
        geolib3::Direction2 toDirection = outEdge.incomingDirection();

        if (turnIsUTurn(fromDirection, toDirection)) {
            penalty = uTurnPenalty;
        } else if (turnIsLeft(fromDirection, toDirection)) {
            if (straightWayIsAvailable || rightTurnAvailable) {
                penalty = leftPenalty;
            } else {
                penalty = 0;
            }
        } else if (turnIsRight(fromDirection, toDirection)) {
            if (straightWayIsAvailable) {
                penalty = rightPenalty;
            } else {
                penalty = 0;
            }
        }
        edge.outEdgesPenalties[i] = penalty;
    }
}

// returns true if edges are connected and the maneuver is allowed
bool RoadNetworkData::edgesAreConnected(EdgeId edge1, EdgeId edge2) const
{
    const Edge& firstEdge = edges_.at(edge1);
    return std::find(firstEdge.outEdges.begin(),
                     firstEdge.outEdges.end(),
                     edge2) != firstEdge.outEdges.end();
}

void RoadNetworkData::deleteEdge(EdgeId edgeId) {
    auto edgeIt = edges_.find(edgeId);
    if (edgeIt == edges_.end()) {
        return;
    }
    // delete edge connections
    for (EdgeId outEdgeId : edgeIt->second.outEdges) {
        auto& outEdge = edges_.at(outEdgeId);
        auto outEdgeToCurEdge = std::find(outEdge.inEdges.begin(),
                                          outEdge.inEdges.end(),
                                          edgeId);
        outEdge.inEdges.erase(outEdgeToCurEdge);
    }
    for (EdgeId inEdgeId : edgeIt->second.inEdges) {
        auto& inEdge = edges_.at(inEdgeId);
        auto inEdgeToCurEdge = std::find(inEdge.outEdges.begin(),
                                         inEdge.outEdges.end(),
                                         edgeId);
        size_t inEdgeToCurEdgeIndex = inEdgeToCurEdge - inEdge.outEdges.begin();
        inEdge.outEdgesPenalties.erase(
            inEdge.outEdgesPenalties.begin() + inEdgeToCurEdgeIndex);
        inEdge.outEdges.erase(inEdgeToCurEdge);
    }
    edges_.erase(edgeIt);
}

void RoadNetworkData::deleteEdges(const std::unordered_set<EdgeId>& edgeIds) {
    for (auto edgeId : edgeIds) {
        deleteEdge(edgeId);
    }
}

PathWithData convertPathToPathWithData(const RoadNetworkData& roadNetwork,
                                       const Path& path)
{
    PathWithData pathWithData;
    for (PathEdge pathEdge : path) {
        pathWithData.push_back({roadNetwork.edge(pathEdge.edgeId),
                                pathEdge.isUsefulAsTarget});
    }
    return pathWithData;
}


std::vector<District> filterDistricts(std::vector<MultiDistrict>& districts,
                                      const boost::optional<MultiDistrict>& mask)
{
    std::vector<District> result;

    for (auto& district : districts) {
        geolib3::MultiPolygon2& area = district.area;
        int counter = 0;
        for (size_t i = 0; i < area.polygonsNumber(); i++) {
            geolib3::Polygon2 p = area.polygonAt(i);
            if (!mask
                || spatialRelation(mask->area, p,
                                   geolib3::SpatialRelation::Intersects)) {
                std::string newName = district.name.name;
                if (counter > 0) {
                    newName += "_" + std::to_string(counter + 1);
                    if (counter == 1) {
                        result.back().name.name += "_1";
                    }
                }
                result.push_back(
                    District{p, DistrictName{newName, district.name.locale}});
                counter++;
            } else {
                INFO() << "Removed district polygon outside mask: "
                       << district.name.name << "\n";
            }
        }

    }
    return result;
}

Meters getPathLength(const RoadNetworkData& roadNetwork,
                     const LoopsPath& path)
{
    Meters length = 0;
    for (const LoopEdge& pathEdge : path) {
        length += roadNetwork.edge(pathEdge.edgeId).length;
    }
    return length;
}

Meters getPathLength(const RoadNetworkData& roadNetwork,
                     const std::vector<EdgeId>& path) {
    Meters length = 0;
    for (EdgeId pathEdge : path) {
        length += roadNetwork.edge(pathEdge).length;
    }
    return length;
}


} // namespace maps::mrc::gen_targets
