#include "easyview_saver.h"

namespace maps::mrc::gen_targets {

geolib3::Segment2 moveToRoadSide2(const geolib3::Segment2& segment, double shift)
{
    double normX = (segment.end().y() - segment.start().y()) / 0.56;
    double normY = segment.start().x() - segment.end().x();
    double l = sqrt(normX * normX + normY * normY);
    normX *= shift / l;
    normY *= shift / l;
    geolib3::Point2 newP1(segment.start().x() + normX / 0.56, segment.start().y() + normY);
    geolib3::Point2 newP2(segment.end().x() + normX / 0.56, segment.end().y() + normY);
    return geolib3::Segment2(newP1, newP2);

}

void EasyViewSaver::saveTarget(const RoadNetworkData& data,
                               const Path& target)
{
    for (size_t i = 0; i < target.size(); i++) {
        if (target[i].isUsefulAsTarget) {
            ostream_ << "!linestyle=rgb(" + getColor(i * 10) + "):2\n";
        } else {
            ostream_ << "!linestyle=rgb(127, 127, 127):2\n";
        }

        auto & points = data.edge(target[i].edgeId).geom.points();
        int back = 0;
        for (size_t j = 1; j < points.size(); j++) {
            double shift = 0.00001 * (1 + 3 * (rand() % 10000) / 10000.0);
            auto segment = moveToRoadSide(points[j-1+back], points[j-back], shift);
            if (j == 1) {
                ostream_ << std::to_string(segment[back].x())
                         << " "
                         << std::to_string(segment[back].y())
                         << " ";
            }
            ostream_ << std::to_string(segment[1-back].x())
                     << " "
                     << std::to_string(segment[1-back].y())
                     << " ";
        }
        ostream_ << "index" + std::to_string(i) + "\n";
    }
}

void EasyViewSaver::drawPolygon(geolib3::Polygon2 polygon,
                                const std::string& label)
{
    ostream_ << "!linestyle=rgb(0, 0, 0):4\n";
    for(size_t i = 0; i < polygon.pointsNumber(); i++) {
        ostream_ << std::to_string(polygon.pointAt(i).x())
                 << " "
                 << std::to_string(polygon.pointAt(i).y())
                 << " ";
    }
    ostream_ << std::to_string(polygon.pointAt(0).x())
             << " "
             << std::to_string(polygon.pointAt(0).y())
             << " ";
    ostream_ << label << "\n";
}

void EasyViewSaver::drawSegments(const std::vector<geolib3::Segment2>& segments,
                                 bool shiftSide, int id)
{
    for(size_t i = 0; i < segments.size(); i++) {
        geolib3::Segment2 segment = segments[i];
        if (shiftSide) {
            double shift = 0.00001 * (1 + 3 * (rand() % 10000) / 10000.0);
            segment = moveToRoadSide2(segment, shift);
        }

        ostream_ << std::to_string(segment.start().x()) << " "
                 << std::to_string(segment.start().y()) << " ";

        if (i == segments.size()-1
            || geolib3::geoDistance(segments[i].end(),
                                    segments[i + 1].start()) > 0.5) {
            ostream_ << std::to_string(segment.end().x()) << " "
                     << std::to_string(segment.end().y()) << " "
                     << "part" << id << "\n";
        }
    }
}

void EasyViewSaver::saveTargets2(RoadNetworkData& data,
                                 std::vector<std::pair<District, Path>>& targets)
{
    for (size_t k = 0; k < targets.size(); k++) {
        auto& target = targets[k].second;
        ostream_ << "!linestyle=rgb(" + getColor(k * 279) + "):2\n";
        std::vector<geolib3::Segment2> segments;
        for (size_t i = 0; i < target.size(); i++) {
            const Edge& edge = data.edge(target[i].edgeId);
            for (size_t j = 0; j < edge.geom.segmentsNumber(); j++) {
                segments.push_back(edge.geom.segmentAt(j));
            }
        }
        drawSegments(segments, true, k);
    }
}


void EasyViewSaver::saveTargets(
    std::vector<std::pair<District, PathWithData>>& targets)
{
    std::set<EdgeId> used;
    for (size_t k = 0; k < targets.size(); k++) {
        auto& target = targets[k].second;
        ostream_ << "!linestyle=rgb(" + getColor(k * 279) + "):2\n";
        for (size_t i = 0; i < target.size(); i++) {
            if (!target[i].isUsefulAsTarget) {
                continue;
                }
            if (used.count(target[i].edge.id)) {
                throw std::runtime_error("Tasks have common edge");
                }

            used.insert(target[i].edge.id);
            auto & points = target[i].edge.geom.points();
            int back = 0;
            for (size_t j = 1; j < points.size(); j++) {
                double shift = 0.00001 * (1 + 3 * (rand() % 10000) / 10000.0);
                auto segment = moveToRoadSide(points[j-1+back], points[j-back], shift);
                if (j == 1) {
                    ostream_ << std::to_string(segment[back].x())
                             << " "
                             << std::to_string(segment[back].y())
                             << " ";
                }
                ostream_ << std::to_string(segment[1-back].x())
                         << " "
                         << std::to_string(segment[1-back].y())
                         << " ";
            }
            ostream_ << "target" + std::to_string(k) + "\n";
        }
    }
}


std::string EasyViewSaver::getColor(size_t index) {
    int r = 0;
    int g = 0;
    int b = 0;
    index %= 256 * 6;
    if (index < 256) {
        r = 255;
        g = index;
    } else if (index < 256 * 2) {
        r = (256 * 2 - 1) - index;
        g = 255;
    } else if (index < 256 * 3) {
        g = 255;
        b = index - (256 * 2);
    } else if (index < 256 * 4) {
        g = (256 * 4 - 1) - index;
        b = 255;
    } else if (index < 256 * 5) {
        b = 255;
        r = index - (256 * 4);
    } else {
        b = (256 * 6 - 1) - index;
        r = 255;
    }

    return std::to_string(r) + ", " + std::to_string(g) + ", "
        + std::to_string(b);
}

std::vector<geolib3::Point2> EasyViewSaver::moveToRoadSide(
    const geolib3::Point2& p1, const geolib3::Point2& p2, double shift)
{
    double normX = (p2.y() - p1.y()) / 0.56;
    double normY = p1.x() - p2.x();
    double l = sqrt(normX * normX + normY * normY);
    normX *= shift / l;
    normY *= shift / l;
    geolib3::Point2 newP1(p1.x() + normX / 0.56, p1.y() + normY);
    geolib3::Point2 newP2(p2.x() + normX / 0.56, p2.y() + normY);
    return {newP1, newP2};

}

} // namespace maps::mrc::gen_targets
