#include "filtered_connected_components.h"

namespace maps::mrc::gen_targets {

namespace {

const int COMPONENT_MINIMAL_SIZE_IN_METERS = 1000;

ComponentId getIdOfTheLargestComponent(
    const std::unordered_map<EdgeId, ComponentId>& components)
{
    std::unordered_map<ComponentId, size_t> sizes;
    size_t largestComponentSize = 0;
    ComponentId largestComponentId = 0;

    for (const auto& edgeComponentPair : components) {
        sizes[edgeComponentPair.second]++;
        if (sizes[edgeComponentPair.second] > largestComponentSize) {
            largestComponentSize = sizes[edgeComponentPair.second];
            largestComponentId = edgeComponentPair.second;
        }
    }
    return largestComponentId;
}

// Filters connected components of targets.
// Returns only big enough components.
std::vector<std::vector<EdgeId>> getFilteredComponents(
    const RoadNetworkData& roadNetwork,
    const std::unordered_map<EdgeId, ComponentId>& edgeIdToComponentId)
{
    int numberOfComponents = 0;
    for (const auto& edgeComponentPair : edgeIdToComponentId) {
        numberOfComponents = std::max(numberOfComponents,
                                      int(edgeComponentPair.second + 1));
    }

    std::vector<std::vector<EdgeId>> resultComponents;
    std::vector<double> resultComponentLengths;
    resultComponents.resize(numberOfComponents);
    resultComponentLengths.resize(numberOfComponents);

    for (auto& edgeComponentPair : edgeIdToComponentId) {
        resultComponents[edgeComponentPair.second].push_back(
            edgeComponentPair.first);
        resultComponentLengths[edgeComponentPair.second] +=
            roadNetwork.edge(edgeComponentPair.first).length;
    }

    for (int i = resultComponents.size() - 1; i >= 0; i--) {
        if (resultComponents[i].size() < 2
            || (resultComponentLengths[i]
                < COMPONENT_MINIMAL_SIZE_IN_METERS)) {
            std::swap(resultComponents[i], resultComponents.back());
            resultComponents.pop_back();
        }
    }
    return resultComponents;
}

} // anonymous namespace

std::vector<std::vector<EdgeId>> getFilteredConnectedComponentsOfTargets(
    const RoadNetworkData& roadNetwork,
    const std::unordered_set<EdgeId>& targetEdges)
{
    std::unordered_map<EdgeId, ComponentId> stronglyConnectedComponents
        = getStronglyConnectedComponents(roadNetwork);
    ComponentId largestComponentId = getIdOfTheLargestComponent(
        stronglyConnectedComponents);

    std::unordered_map<EdgeId, ComponentId> targetConnectedComponents
        = getWeaklyConnectedComponents(
            roadNetwork,
            [&] (EdgeId edgeId) {
                return targetEdges.count(edgeId)
                       && stronglyConnectedComponents[edgeId] == largestComponentId;
            });

    return getFilteredComponents(roadNetwork,
                                 targetConnectedComponents);
}

std::unordered_set<EdgeId> getEdgesInClosedAreas(const RoadNetworkData& roadNetwork)
{
    std::unordered_map<EdgeId, ComponentId> stronglyConnectedComponents
        = getStronglyConnectedComponents(roadNetwork);
    ComponentId largestComponentId = getIdOfTheLargestComponent(
        stronglyConnectedComponents);
    std::unordered_set<EdgeId> edgesInClosedAreas;
    for (const auto& edge : roadNetwork.edges()) {
        if (stronglyConnectedComponents[edge.first] != largestComponentId) {
            edgesInClosedAreas.insert(edge.first);
        }
    }
    return edgesInClosedAreas;
}

} // namespace maps::mrc::gen_targets
