#include "geojson.h"

#include "task_area.h"

#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/deprecated/localeutils/include/locale.h>
#include <maps/libs/log8/include/log8.h>

#include <boost/filesystem.hpp>

#include <fstream>

namespace bf = boost::filesystem;

namespace maps::mrc::gen_targets {

const double JSON_TARGET_MIN_LENGTH_THRESHOLD = 5; // meters

std::vector<MultiDistrict> readPolygonsFromJsonFile(const std::string& path)
{
    auto json = maps::json::Value::fromFile(path);

    std::vector<MultiDistrict> results;

    int i = 1;

    for (auto& feature : json["features"]) {
        std::string name = feature["properties"]["description"].as<std::string>("");
        if (feature["geometry"]["type"].as<std::string>("") != "Polygon") {
            continue;
        }

        if (name == "") {
            name = "area_" + std::to_string(i);
        } else {
            name.erase(std::remove(name.begin(), name.end(), '\n'), name.end());
        }
        i++;

        bool allPointsAreSimilar = true;

        geolib3::PointsVector points;
        for (auto& coord : feature["geometry"]["coordinates"][0]) {
            double lon = coord[0].as<double>();
            double lat = coord[1].as<double>();
            if (points.size()
                && (points.back().x() != lon || points.back().y() != lat)) {
                allPointsAreSimilar = false;
            }
            points.emplace_back(lon, lat);
        }
        if (allPointsAreSimilar) {
            WARN() << "Skipped broken geojson polygon with zero area";
            continue;
        }
        results.push_back(
            MultiDistrict{geolib3::MultiPolygon2(
                    std::vector<geolib3::Polygon2>{geolib3::Polygon2(points)}),
            DistrictName{name, Locale(LangCode::LANG_RU)}});
    }

    return results;
}

std::vector<geolib3::Polyline2> readPolylinesFromJsonFile(const std::string& path)
{
    auto json = maps::json::Value::fromFile(path);

    std::vector<geolib3::Polyline2> results;

    int i = 1;

    for (auto& feature : json["features"]) {
        if (feature["geometry"]["type"].as<std::string>("") != "LineString") {
            continue;
        }

        i++;

        geolib3::PointsVector points;
        for (auto& coord : feature["geometry"]["coordinates"]) {
            double lon = coord[0].as<double>();
            double lat = coord[1].as<double>();
            points.emplace_back(lon, lat);
        }
        results.push_back(geolib3::Polyline2(points));
    }
    return results;
}

std::vector<bf::path> listGeojsonFilesInDir(const std::string& dirPath) {
    std::vector<bf::path> files;
    bf::path path(dirPath);
    for (auto it = bf::directory_iterator(path);
         it != bf::directory_iterator();
         it++) {
        if (bf::is_directory(it->status()))
            continue;
        if (it->path().extension().string() != ".geojson")
            continue;
        files.push_back(*it);
    }
    return files;
}

std::pair<std::vector<std::pair<District, Path>>, RoadNetworkData>
createTasksFromGeojsons(const std::string& dirPath) {
    std::vector<std::pair<District, Path>> tasks;
    std::vector<bf::path> filenames = listGeojsonFilesInDir(dirPath);
    Edges edges;
    EdgeId nextId = 1;
    for (bf::path filename : filenames) {
        std::vector<geolib3::Polyline2>
            targets = readPolylinesFromJsonFile(filename.string());
        Path path;
        for (geolib3::Polyline2 target : targets) {
            if (geolib3::geoLength(target) < JSON_TARGET_MIN_LENGTH_THRESHOLD) {
                continue;
            }
            edges[nextId].id = nextId;
            edges[nextId].geom = target;
            edges[nextId].fc = 1;
            edges[nextId].length = geolib3::geoLength(target);
            edges[nextId].isTarget = true;
            path.push_back(PathEdge{nextId, true});
            nextId += 1;
        }
        if (!path.size()) {
            continue;
        }
        std::pair<District, Path> task;
        task.second = path;
        task.first.name.name = filename.stem().string();
        auto edgesCopy = edges;
        correctTaskArea(task, RoadNetworkData(std::move(edgesCopy)));
        tasks.push_back(task);
    }

    return std::make_pair(tasks, RoadNetworkData(std::move(edges)));
}


std::string createGeojsonPolygon(const std::string& name, size_t id, std::vector<geolib3::Point2> points) {
    std::string result;
    result = "{\"type\":\"Feature\",\"id\":" + std::to_string(id)
        + ",\"geometry\":{\"type\":\"Polygon\",\"coordinates\":[[";
    for (auto& point : points) {
        result += "[" + std::to_string(point.x()) + "," + std::to_string(point.y()) + "],";
    }
    result.pop_back();
    result += "]]},\"properties\":{\"description\":\"" + name
        +"\",\"fill\":\"#ed4543\",\"fill-opacity\":0.6,\"stroke\":\"#ed4543\",\"stroke-width\":\"5\",\"stroke-opacity\":0.9}}";
    return result;
}

void savePolygonsToJsonFile(std::vector<MultiDistrict> multiDisctricts, const std::string& path)
{
    std::string jsonStr;
    jsonStr ="{\"type\":\"FeatureCollection\",\"metadata\":{\"name\":\"Generated districts\"},\"features\":[";
    for (auto& multiDisctrict : multiDisctricts) {
        for (size_t i = 0; i < multiDisctrict.area.polygonsNumber(); i++) {
            std::string name = multiDisctrict.name.name;
            if (multiDisctrict.area.polygonsNumber() > 1) {
                name += "_" + std::to_string(i+1);
            }
            const geolib3::Polygon2& polygon = multiDisctrict.area.polygonAt(i);
            std::vector<geolib3::Point2> points;
            for (size_t j = 0; j < polygon.pointsNumber(); j++) {
                points.push_back(polygon.pointAt(j));
            }
            jsonStr += createGeojsonPolygon(name, i, points) + ",";
        }
    }
    jsonStr.pop_back();
    jsonStr += "]}";

    std::ofstream outFile(path);
    outFile << jsonStr;
}

std::string createGeojsonPolyline(const std::string& name,
                                  size_t& id,
                                  const std::vector<geolib3::Point2>& points) {
    std::string result;
    result = "{\"type\":\"Feature\",\"id\":" + std::to_string(id)
        + ",\"geometry\":{\"type\":\"LineString\",\"coordinates\":[";
    for (auto& point : points) {
        result += "[" + std::to_string(point.x()) + "," + std::to_string(point.y()) + "],";
    }
    result.pop_back();
    result += "]},\"properties\":{\"description\":\"" + name
        +"\",\"stroke\":\"#ed4543\",\"stroke-width\":5,\"stroke-opacity\":0.9}},";
    id++;
    return result;
}


void saveTasksToJsonFile(std::vector<std::pair<District, PathWithData>>& tasks,
                         const std::string& path)
{
    std::string jsonStr;
    size_t id = 0;
    jsonStr ="{\"type\":\"FeatureCollection\",\"metadata\":{\"name\":\"Generated districts\"},\"features\":[";

    for (const auto& task : tasks) {
        for (const PathEdgeWithData& pathEdge : task.second) {
            jsonStr += createGeojsonPolyline(task.first.name.name, id,
                                             pathEdge.edge.geom.points());
            id++;
        }

    }
    if (jsonStr.back() == ',') {
        jsonStr.pop_back();
    }
    jsonStr +="]}";

    std::ofstream outFile(path);
    outFile << jsonStr;
}

} // namespace maps::mrc::gen_targets
