#include "path_search.h"

#include <maps/libs/log8/include/log8.h>

#define _USE_MATH_DEFINES
#include <cmath>

namespace maps::mrc::gen_targets {

namespace {

// PathSearch tries to search the path not using penalty edges. If it
// is impossible, it uses all edges. The larger this value the larger the
// area of non-penalty searching
const double PENALTY_MULTIPLIER = 1000000000;

} // namespace

PathSearch::PathSearch(const RoadNetworkData& roadNetwork,
                       EdgeId startEdge,
                       const std::unordered_set<EdgeId>& finishEdges,
                       const std::unordered_set<EdgeId>& penaltyEdges)
    : roadNetwork_(roadNetwork)
    , penaltyEdges_(penaltyEdges)
    , startEdge_(startEdge)
{
    Step firstStep;
    firstStep.curEdge = startEdge;
    firstStep.distanceFromStart = 0;
    firstStep.stepsFromStart = 0;
    firstStep.stepsFromPrevTurn = 1;
    applyStep(firstStep);

    while (true) {
        if (queue_.empty()) {
            throw CantFindPathException("can't find path");
        }
        auto currentStep = queue_.top();
        queue_.pop();

        if (!visitedEdges_.insert(currentStep.curEdge).second) {
            continue;
        }
        prevEdge_[currentStep.curEdge] = currentStep.prevEdge;
        if (finishEdges.count(currentStep.curEdge)) {
            finishEdge_ = currentStep.curEdge;
            finishSteps_ = currentStep.stepsFromStart;
            break;
        }
        applyStep(currentStep);
    }
}

void PathSearch::applyStep(const Step& curStep)
{
    auto waysAndPenalties = roadNetwork_.getOutEdgesAndPenalties(curStep.curEdge);
    auto& ways = waysAndPenalties.first;
    auto& waysTurnPenalties = waysAndPenalties.second;

    for (size_t i = 0; i < ways.size(); i++) {
        const Edge& wayEdge = ways[i].get();
        if (visitedEdges_.count(wayEdge.id)) {
            continue;
        }

        auto& curEdge = roadNetwork_.edge(curStep.curEdge);
        Seconds edgeDrivingTime = curEdge.time;
        Seconds turningTime = waysTurnPenalties[i];
        if (!curEdge.isTarget
            || penaltyEdges_.count(curEdge.id)) {
            edgeDrivingTime *= PENALTY_MULTIPLIER;
            turningTime *= PENALTY_MULTIPLIER;
        }
        Seconds unrecommendedTurnsPenalty
            = roadNetwork_.getUnrecommendedManeuversPenalty(curStep.curEdge,
                                                            wayEdge.id);
        bool wayIsTarget = true;
        if (!wayEdge.isTarget
            || penaltyEdges_.count(wayEdge.id)) {
            wayIsTarget = false;
        }

        // The longer the straight road before the turn is, the
        // smaller penalty is.
        // E.g. there is a long straight target road with several U turns.
        // The path starts and ends at the beginning of the road.
        // The longest route (with the last uturn) is the best for us.
        // With turningTime modifier it will have the smallest penalty
        turningTime /= curStep.stepsFromPrevTurn;

        Step step;
        step.curEdge = wayEdge.id;
        step.prevEdge = curStep.curEdge;
        if (turningTime == 0) {
            step.stepsFromPrevTurn = curStep.stepsFromPrevTurn;
            if (wayIsTarget) {
                step.stepsFromPrevTurn++;
            }
        } else {
            step.stepsFromPrevTurn = 1;
        }
        step.distanceFromStart = curStep.distanceFromStart + edgeDrivingTime
            + turningTime + unrecommendedTurnsPenalty;
        step.stepsFromStart = curStep.stepsFromStart + 1;

        queue_.push(step);
    }
}

std::vector<EdgeId> PathSearch::getResult()
{
    std::vector<EdgeId> result;
    result.resize(finishSteps_);
    EdgeId curEdge = finishEdge_;
    for (int i = 0; i < finishSteps_; i++) {
        result[i] = curEdge;
        curEdge = prevEdge_[curEdge];
    }
    std::reverse(result.begin(), result.end());

    REQUIRE(roadNetwork_.edgesAreConnected(startEdge_, result.front()),
            "Error in path_search algorithm, the path is not connected to the start edge");
    return result;
}

} // namespace maps::mrc::gen_targets
