#pragma once

#include "data_model.h"

#include <maps/libs/common/include/exception.h>

#include <map>
#include <set>
#include <queue>
#include <unordered_set>
#include <utility>

namespace maps::mrc::gen_targets {

class CantFindPathException : public maps::Exception {
    using Exception::Exception;
};

// step of the Dijkstra algorithm where we found the shortest way to
// curEdge. We mark curEdge as visited and observe its output edges.
struct Step {
    EdgeId curEdge;
    EdgeId prevEdge;
    int stepsFromStart;
    Seconds distanceFromStart;
    int stepsFromPrevTurn;

    bool operator>(const Step& other) const
    {
        if (distanceFromStart == other.distanceFromStart) {
            return stepsFromStart > other.stepsFromStart;
        }
        return distanceFromStart > other.distanceFromStart;
    }
};

// @brief Dijkstra path searching algorithm.
class PathSearch {
public:
    // @brief finds the shortest path from startEdge to any edge from
    // finishEdges.
    // Distance penalties is consist of edges lenghts
    // and turns penalties.
    // Target edges have smaller penalties (if they are not listed in
    // penaltyEdges set)
    PathSearch(const RoadNetworkData& roadNetwork,
               EdgeId startEdge,
               const std::unordered_set<EdgeId>& finishEdges,
               const std::unordered_set<EdgeId>& penaltyEdges);
    std::vector<EdgeId> getResult();

private:
    void applyStep(const Step& cur);

    // @brief returns turn penalty for each way. Penalty is depend on
    // graph geometry and number of adjacent targets
    std::vector<Seconds> getTurnsPenalties(const Edge& curEdge,
                                          const EdgeCRefVec& ways);

private:
    const RoadNetworkData& roadNetwork_;
    const std::unordered_set<EdgeId>& penaltyEdges_;
    EdgeId startEdge_;
    EdgeId finishEdge_;
    int finishSteps_;
    std::unordered_map<EdgeId, EdgeId> prevEdge_;
    std::unordered_set<EdgeId> visitedEdges_;
    std::priority_queue<Step, std::vector<Step>, std::greater<Step>> queue_;
};

} // namespace maps::mrc::gen_targets
