#pragma once

#include "path_search.h"
#include "split_path.h"

#include <list>
#include <iostream>

// Algorigthm that generates one loop route that covers all the target
// edges.
// https://wiki.yandex-team.ru/jandekskarty/projects/catalog/ek/ek-nk/projects/automnyak/main/Algoritm-generatora-marshrutov-obezda-gorodov/#sozdanieciklicheskogomarshrutaproxodjashhegopovsemceljam
//
// Input data: a graph and a list of target edges. All the target
// edges should be in one strongly connected component in graph. The
// subgraph that consists only of targets should be a weakly connected component.
//
// Algorithm description:
// 1) take some target edge - the init edge
// 2) find the shortest path from the init edge to itself - current loop
// 3) mark all the edges of the current loop as visited
// 4) discover each edge of the current loop:
//      Assume we are discovering edge A and the next edge in the loop
//      is edge B. If edge A has unvisited output target edges, we
//      take one of that edges(e.g. edge C) and find the shortest path
//      from edge C to edge B (e.g. C->D->E->B). Then we take the
//      current loop and replace A->B with A->C->D->E->B. Now the
//      current loop is still connected and covers more target edges.
//      (Notice that added path C->D->E is also looks like a loop, but
//      we don't know if maneuver E->C is allowed. Such small loops can
//      be usefull for task splitting)
// 5) repeat steps 3,4 while current loop edges have unvisited target
// output edges
// 6) now the current loop covers all the target edges.
//
// To minimize route overhead we use smart path searching
// algorithm in step 4: good paths should have more uncovered target
// edges and less non-target edges and covered target edges
//
// for step 4 we use queue of possible turns and applies better turns first

namespace maps::mrc::gen_targets {

struct PossibleTurn {
    // pointer to the loop edge (edge A in the algorithm description)
    std::list<LoopEdge>::iterator edgeBeforeTurn;

    EdgeId curNextEdgeinLoop; // (edge B in the algorighm description)
    EdgeId edgeAfterTurn; // (edge C in the algorithm description)
    double maneuverPenaltyChange; // (A->C turn penalty) - (A->B turn penalty)

    // (A->C unrecommendedManeuvers penalty) - (A->B unrecommendedManeuvers penalty)
    // (see unrecommended_maneuvers.h)
    double unrecommendedManeuverPenaltyChange;

    // number of unvisited target input edges of edge C. If edge C has
    // unvisited target input edges, maybe we don't need to turn from
    // edge A to edge C because we can reach edge C from other target
    // edges. If edge C doesn't have unvisited target input edges,
    // then we should turn to from edge A to edge C because it is the
    // only possibility to visit edge C
    int unvisitedInputs;

    // each step 4 creates a possible loop in the route. We will use
    // these loops in task splitting. It is hard to split path with
    // very long loop so we will try not to extend already long
    // loops(i.e. we will try not to apply possible turns from edges
    // of long loops)
    double loopLength; // length of the path created on step 4

    // amount of overhead edges that will be added when applying this
    // possible turn.
    // = (sum of overhead edges time) / (sum of non-overhead edges time)
    double overheadRatio;

    // the order of the possible turns
    bool operator>(const PossibleTurn& other) const;
};

class RouteGenerator {
public:
    RouteGenerator(const RoadNetworkData& roadNetwork, EdgeId initEdge);
    // possible loop that was created on each step 4
    LoopsPath getResult();

private:
    void initialize();
    void createInitLoop(EdgeId initEdgeId);
    void generate();

    // after creating a possibleTurn current loop may be changed and
    // "edge C" of the possible turn may be already visited
    bool possibleTurnIsActual(const PossibleTurn& possibleTurn);
    void applyPossibleTurn(PossibleTurn possibleTurn);
    void handleNewEdgeInLoop(std::list<LoopEdge>::iterator loopIt);
    void addPossibleTurn(std::list<LoopEdge>::iterator edgeBeforeTurn,
                         EdgeId edgeAfterTurn);

    // Returns (sum of overhead edges time) / (sum of all edges time).
    // Full path = initEdge + path
    // Returns overhead on the current generator stage(i.e. includes
    // currently visited edges)
    double getPathOverheadRatio(EdgeId initEdge, const std::vector<EdgeId>& path);

    // recalculate all the possible turns from the current loop to the
    // edgeAfterTurn edge
    void recalculateTurns(EdgeId edgeAfterTurn);
    std::list<LoopEdge>::iterator nextEdgeInLoop(std::list<LoopEdge>::iterator loopIt);

private:
    const RoadNetworkData& roadNetwork_;
    std::priority_queue<PossibleTurn,
                        std::vector<PossibleTurn>,
                        std::greater<PossibleTurn>> queue_;

    // number of unvisited target input edges of the edge
    std::unordered_map<EdgeId, int> unvisitedEdgeInputs_;
    std::list<LoopEdge> loop_; // current loop
    std::unordered_set<EdgeId> visitedEdges_;

    // for each edge we store the pointers to all the current loop
    // edges which have possible turn to the edge
    std::unordered_map<EdgeId, std::vector<std::list<LoopEdge>::iterator>> visitedEdgeInputs_;
    LoopId nextFreeLoopId_; // LoopID generator
    std::unordered_map<LoopId, double> loopLength_; // length of each
                                                    // possible loop
    double targetsLength_;
    double generatedLength_;
};

} // namespace maps::mrc::gen_targets
