#include "unrecommended_maneuvers.h"

#include <queue>
#include <unordered_set>

#include <boost/range/algorithm_ext/erase.hpp>

namespace maps::mrc::gen_targets {

UnrecommendedManeuversCalculator::UnrecommendedManeuversCalculator(
    const ExtendedRoadNetwork& roadNetwork)
    : roadNetwork_(roadNetwork)
    , edges_(roadNetwork.getEdges())
{
    calculateUnrecommendedManeuvers();
}

std::unordered_map<EdgeId, std::vector<EdgeId>>
UnrecommendedManeuversCalculator::getUnrecommendedManeuvers()
{
    return unrecommendedManeuvers_;
}


// If some edge1 is target and has only one input target edge - edge2,
// then maneuver edge2->edge1 is recommended and other maneuvers
// from edge2 are unrecommended.
// If some edge1 is target and has only one output target edge - edge2,
// then maneuver edge1->edge2 is recommended and other maneuvers
// to edge2 are unrecommended.
void UnrecommendedManeuversCalculator::calculateUnrecommendedManeuvers()
{
    for (const auto& node : roadNetwork_.nodes()) {
        bool changed = true;
        while(changed) {
            changed = false;
            for (EdgeId curEdgeId : node.second.inEdges) {
                const Edge& curEdge = edges_[curEdgeId];
                std::vector<EdgeId> outTargets = getTargets(curEdge.outEdges);
                if (curEdge.isTarget && outTargets.size() == 1) {
                    const Edge& outTarget = edges_[outTargets[0]];
                    if (getTargets(outTarget.inEdges).size() > 1) {
                        setSingleInputEdge(outTarget.id, curEdgeId);
                        changed = true;
                    }
                }
            }
            for (EdgeId curEdgeId : node.second.outEdges) {
                const Edge& curEdge = edges_[curEdgeId];
                std::vector<EdgeId> inTargets = getTargets(curEdge.inEdges);
                if (curEdge.isTarget && inTargets.size() == 1) {
                    const Edge& inTarget = edges_[inTargets[0]];
                    if (getTargets(inTarget.outEdges).size() > 1) {
                        setSingleOutputEdge(inTarget.id, curEdgeId);
                        changed = true;
                    }
                }
            }

        }
    }
}

std::vector<EdgeId> UnrecommendedManeuversCalculator::getTargets(
    const std::vector<EdgeId> edgeIds)
{
    std::vector<EdgeId> targets;
    for (EdgeId edgeId : edgeIds) {
        if (edges_[edgeId].isTarget) {
            targets.push_back(edgeId);
        }
    }
    return targets;
}

void UnrecommendedManeuversCalculator::setSingleOutputEdge(EdgeId edgeId,
                                                           EdgeId singleOutEdgeId)
{
    Edge& edge = edges_[edgeId];
    for (EdgeId outEdgeId : edge.outEdges) {
        if (outEdgeId == singleOutEdgeId) {
            continue;
        }
        addUnrecommendedManeuver(edge.id, outEdgeId);
        boost::remove_erase_if(edges_[outEdgeId].inEdges, [&edge](EdgeId edgeId) {
                return edgeId == edge.id;
            });
    }
    edge.outEdges.resize(1);
    edge.outEdges[0] = singleOutEdgeId;
}

void UnrecommendedManeuversCalculator::setSingleInputEdge(EdgeId edgeId,
                                                          EdgeId singleInEdgeId)
{
    Edge& edge = edges_[edgeId];
    for (EdgeId inEdgeId : edge.inEdges) {
        if (inEdgeId == singleInEdgeId) {
            continue;
        }
        addUnrecommendedManeuver(inEdgeId, edge.id);
        boost::remove_erase_if(edges_[inEdgeId].outEdges, [&edge](EdgeId edgeId) {
                return edgeId == edge.id;
            });

    }
    edge.inEdges.resize(1);
    edge.inEdges[0] = singleInEdgeId;
}

void UnrecommendedManeuversCalculator::addUnrecommendedManeuver(EdgeId from,
                                                                EdgeId to)
{
    unrecommendedManeuvers_[from].push_back(to);
}

} // namespace maps::mrc::gen_targets
