#include "graph_creator.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/filtered_connected_components.h>

namespace maps::mrc::gen_targets::tests {

TEST(filtered_connected_components_tests, test_one_component)
{
    //  graph:
    //     --5->
    //     ^    |
    //     |    6
    //     4    |
    //     |    V
    //--1-> <-2-- --3->

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {4})
        .addEdge(2, {2, 0}, {1, 0}, {4})
        .addEdge(3, {2, 0}, {3, 0}, {})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {2, 3});

    RoadNetworkData roadNetwork = graphCreator.createGraph();
    {
        auto components = getFilteredConnectedComponentsOfTargets(
            roadNetwork, {1, 2, 3, 4, 5, 6});
        EXPECT_EQ(components.size(), 1u);
        std::set<EdgeId> result(components[0].begin(), components[0].end());
        EXPECT_THAT(result, testing::UnorderedElementsAre(2, 4, 5, 6));
    }

    {
        // if edge 6 is not target, component will still be connected
        auto components = getFilteredConnectedComponentsOfTargets(
            roadNetwork, {2, 4, 5});
        EXPECT_EQ(components.size(), 1u);
        std::set<EdgeId> result(components[0].begin(), components[0].end());
        EXPECT_THAT(result, testing::UnorderedElementsAre(2, 4, 5));
    }
}

TEST(filtered_connected_components_tests, test_two_disconnected_components)
{
    //  graph:
    // --5->       --9-->
    // ^    |      ^    |
    // |    6       \   10
    // 4    |        8  |
    // |    v         \ v
    // <-2--

    // Take only the largest strongly connected component
    // We consider that all the small components
    // are closed areas

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {4})
        .addEdge(2, {2, 0}, {1, 0}, {4})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {2})
        .addEdge(8, {4, 0}, {3, 1}, {9})
        .addEdge(9, {3, 1}, {4, 1}, {10})
        .addEdge(10, {4, 1}, {4, 0}, {8});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto components = getFilteredConnectedComponentsOfTargets(
        roadNetwork, {2, 4, 5, 6, 8, 9, 10});
    EXPECT_EQ(components.size(), 1u);
    std::set<EdgeId> result(components[0].begin(), components[0].end());
    EXPECT_THAT(result, testing::UnorderedElementsAre(2, 4, 5, 6));

}

TEST(filtered_connected_components_tests, test_barrier_area)
{
    //  graph:
    // --5->       --9-->
    // ^    |      ^   /
    // |    6      |  10
    // 4    |      8 /
    // |    v      |v
    // <-2-- --3->

    // Components are connected, but not strongly
    // Take only the largest component
    // We consider that all the small components
    // are closed areas

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {4})
        .addEdge(2, {2, 0}, {1, 0}, {4})
        .addEdge(3, {2, 0}, {3, 0}, {8})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {2, 3})
        .addEdge(8, {3, 0}, {3, 1}, {9})
        .addEdge(9, {3, 1}, {4, 1}, {10})
        .addEdge(10, {4, 1}, {3, 0}, {8});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto components = getFilteredConnectedComponentsOfTargets(
        roadNetwork, {2, 3, 4, 5, 6, 8, 9, 10});
    EXPECT_EQ(components.size(), 1u);
    std::set<EdgeId> result(components[0].begin(), components[0].end());
    EXPECT_THAT(result, testing::UnorderedElementsAre(2, 4, 5, 6));
}

TEST(filtered_connected_components_tests, test_barrier_area2)
{
    //  graph:
    // ----------------------
    // |                    |
    // | <-6--<-5--<--4--   |
    // | --1->--2->|--3->   |
    // |                    |
    // |  polygon           |
    // ----------------------

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2})
        .addEdge(2, {1, 0}, {2, 0}, {5})
        .addEdge(5, {2, 0}, {1, 0}, {6})
        .addEdge(6, {1, 0}, {0, 0}, {1})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {2, 0}, {3, 5});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto components = getFilteredConnectedComponentsOfTargets(
        roadNetwork, {1, 2, 3, 4, 5, 6});
    EXPECT_EQ(components.size(), 1u);
    std::set<EdgeId> result(components[0].begin(), components[0].end());
    EXPECT_THAT(result, testing::UnorderedElementsAre(1, 2, 5, 6));
}

TEST(filtered_connected_components_tests, test_two_connected_components)
{
    //  graph:
    // --5-> <-7-- --9-->
    // ^    |      ^   /
    // |    6      |  10
    // 4    |      8 /
    // |    v      |v
    // <-2-- --3->

    // edges 3 and 7 are not targets

    // There are two separate strongly connected components
    // of target
    // We should generate tasks for each component

    GraphCreator graphCreator;
    graphCreator
        .addEdge(2, {2, 0}, {1, 0}, {4})
        .addEdge(3, {2, 0}, {3, 0}, {8}, false)
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {2, 3})
        .addEdge(7, {3, 1}, {2, 1}, {6}, false)
        .addEdge(8, {3, 0}, {3, 1}, {7, 9})
        .addEdge(9, {3, 1}, {4, 1}, {10})
        .addEdge(10, {4, 1}, {3, 0}, {8});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto components = getFilteredConnectedComponentsOfTargets(
        roadNetwork, {2, 4, 5, 6, 8, 9, 10});
    EXPECT_EQ(components.size(), 2u);
    std::set<EdgeId> result1(components[0].begin(), components[0].end());
    std::set<EdgeId> result2(components[1].begin(), components[1].end());
    if (result1 == std::set<EdgeId>{2, 4, 5, 6}) {
        EXPECT_THAT(result2, testing::UnorderedElementsAre(8, 9, 10));
    } else {
        EXPECT_THAT(result1, testing::UnorderedElementsAre(8, 9, 10));
        EXPECT_THAT(result2, testing::UnorderedElementsAre(2, 4, 5, 6));
    }
}

TEST(filtered_connected_components_tests, test_one_big_component)
{
    //  graph:
    // --5-> <-11- --9-->
    // ^    |      ^    |
    // |    6      |    10
    // 4    |      8    |
    // |    v      |    v
    // <-2-- --3-> <-7--

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {4})
        .addEdge(2, {2, 0}, {1, 0}, {4})
        .addEdge(3, {2, 0}, {3, 0}, {8})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {2, 3})
        .addEdge(7, {4, 0}, {3, 0}, {8})
        .addEdge(8, {3, 0}, {3, 1}, {9, 11})
        .addEdge(9, {3, 1}, {4, 1}, {10})
        .addEdge(10, {4, 1}, {4, 0}, {7})
        .addEdge(11, {3, 1}, {2, 1}, {6});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto components = getFilteredConnectedComponentsOfTargets(
        roadNetwork, {2, 3, 4, 5, 6, 7, 8, 9, 10});
    EXPECT_EQ(components.size(), 1u);
    std::set<EdgeId> result(components[0].begin(), components[0].end());
    EXPECT_THAT(result, testing::UnorderedElementsAre(2, 3, 4, 5, 6, 7, 8, 9, 10));
}

TEST(filtered_connected_components_tests, test_complicated_components)
{
    //  graph:
    // --5-> <-11- --9-->
    // ^    |      ^    |
    // |    6      |    10
    // 4    |      8    |
    // |    v      |    v    --12->
    // <-2-- --3-> <-7--     <-13--

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {4})
        .addEdge(2, {2, 0}, {1, 0}, {4})
        .addEdge(3, {2, 0}, {3, 0}, {8})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {2, 3})
        .addEdge(7, {4, 0}, {3, 0}, {8})
        .addEdge(8, {3, 0}, {3, 1}, {9, 11})
        .addEdge(9, {3, 1}, {4, 1}, {10})
        .addEdge(10, {4, 1}, {4, 0}, {7})
        .addEdge(11, {3, 1}, {2, 1}, {6})
        .addEdge(12, {4, 0}, {5, 0}, {13})
        .addEdge(13, {5, 0}, {4, 0}, {12});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto components = getFilteredConnectedComponentsOfTargets(
        roadNetwork, {2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 13});
    EXPECT_EQ(components.size(), 1u);
    std::set<EdgeId> result(components[0].begin(), components[0].end());
    EXPECT_THAT(result, testing::UnorderedElementsAre(2, 3, 4, 5, 6, 7, 8, 9, 10));
}

TEST(filtered_connected_components_tests, test_many_components)
{
    //  graph:
    //
    //  <-----5--     <--4--    <-8--
    //  ------1->--2-->--3->    --7->

    // we should select only the biggest strongly connected component

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {-1, 0}, {1, 0}, {2, 5})
        .addEdge(2, {1, 0}, {2, 0}, {3})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {2, 0}, {3,})
        .addEdge(5, {1, 0}, {0, 0}, {1})
        .addEdge(7, {4, 0}, {5, 0}, {8})
        .addEdge(8, {5, 0}, {4, 0}, {7});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto components = getFilteredConnectedComponentsOfTargets(
        roadNetwork, {1, 2, 3, 4, 5, 7, 8});
    EXPECT_EQ(components.size(), 1u);
    EXPECT_THAT(components[0],
                testing::UnorderedElementsAre(1, 5));
}

} // namespace maps::mrc::gen_targets::tests
