#pragma once

#include "maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/data_model.h"

#include <maps/libs/geolib/include/vector.h>

namespace maps::mrc::gen_targets::tests {

namespace {

const double KILOMETER = 1000.0;
const geolib3::Point2 P0(37.61, 55.75); // Moscow center
const geolib3::Vector2 DX = fastGeoShift(P0, geolib3::Vector2(KILOMETER, 0)) - P0;
const geolib3::Vector2 DY = fastGeoShift(P0, geolib3::Vector2(0, KILOMETER)) - P0;

}

using Kilometers = double;
using XY = std::pair<Kilometers, Kilometers>;

// converts Cartesian coordiantes into lon lat
// xy(0, 0) - is the Moscow center
inline geolib3::Point2 xyToLonLat(XY xy) { return P0 + xy.first * DX + xy.second * DY; }

// Creates graph in Moscow center using Cortesian
// coordinates(in kilometers) of edges
class GraphCreator {
public:
    GraphCreator& addEdge(EdgeId edgeId,
                          XY beginCoordinates,
                          XY endCoordinates,
                          const std::vector<EdgeId>& outEdges,
                          bool isTarget = true) {
        Edge edge;
        edge.id = edgeId;
        edge.fc = 1;
        edge.isUTurn = false;
        edge.isTarget = isTarget;
        edge.geom = geolib3::Polyline2(
            geolib3::PointsVector{xyToLonLat(beginCoordinates),
                                  xyToLonLat(endCoordinates)});
        edge.outEdges = outEdges;
        edge.length = geolib3::geoLength(edge.geom);
        edge.time = edge.length / 10;

        edges_[edgeId] = edge;
        return *this;
    };

    GraphCreator& setEdgeIsTarget(EdgeId edgeId, bool isTarget) {
        edges_[edgeId].isTarget = isTarget;
        return *this;
    }

    GraphCreator& setAllEdgesIsTarget(bool isTarget) {
        for (auto& edge : edges_) {
            edge.second.isTarget = isTarget;
        }
        return *this;
    }

    RoadNetworkData createGraph() {
        auto edgesCopy = edges_;
        for (auto& it : edgesCopy) {
            Edge& edge = it.second;
            for (auto outEdge : edge.outEdges) {
                edgesCopy[outEdge].inEdges.push_back(it.first);
            }
        }
        return RoadNetworkData(std::move(edgesCopy));
    }

    std::unordered_set<EdgeId> getTargetEdges() {
        std::unordered_set<EdgeId> result;
        for (auto& it : edges_) {
            result.insert(it.first);
        }
        return result;
    }

public:
    Edges edges_;
};

inline geolib3::Polygon2 createPolygon(const std::vector<XY>& polygonBorder) {
    std::vector<geolib3::Point2> lonLatPolygonBorder;
    for (XY point : polygonBorder) {
        lonLatPolygonBorder.push_back(xyToLonLat(point));
    }
    return geolib3::Polygon2(lonLatPolygonBorder);
}

} // namespace maps::mrc::gen_targets::tests
