#include "graph_creator.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/path_search.h>

namespace maps::mrc::gen_targets::tests {

TEST(path_search, test_simple_search_without_crossroads)
{
    // graph:
    // --1->--2->--3->
    //

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2})
        .addEdge(2, {1, 0}, {2, 0}, {3})
        .addEdge(3, {2, 0}, {3, 0}, {});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // search the path from edge1 to edge3 where all the edges are targets
    auto path = PathSearch(roadNetwork, 1, {3}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{2, 3}));

    graphCreator
        .setEdgeIsTarget(1, false)
        .setEdgeIsTarget(2, false)
        .setEdgeIsTarget(3, false);
    roadNetwork = graphCreator.createGraph();

    // search the path from edge1 to edge3 where all the edges are not targets
    path = PathSearch(roadNetwork, 1, {3}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{2, 3}));
}

TEST(path_search, test_cant_find_path)
{
    // graph:
    // --1->--2->--3->
    //

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2})
        .addEdge(2, {1, 0}, {2, 0}, {3})
        .addEdge(3, {2, 0}, {3, 0}, {});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // no paths from edge2 to edge1. PathSearch should throw an exception
    EXPECT_THROW(PathSearch(roadNetwork, 2, {1}, {}), CantFindPathException);

    // no paths from edge2 to itself. PathSearch should throw an exception
    EXPECT_THROW(PathSearch(roadNetwork, 2, {2}, {}), CantFindPathException);

}

TEST(path_search, test_search_two_routes)
{
    //  graph:
    //     --5->
    //     ^    |
    //     |    6
    //     4    |
    //     |    V
    // --1->--2->--3->

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 4})
        .addEdge(2, {1, 0}, {2, 0}, {3})
        .addEdge(3, {2, 0}, {3, 0}, {})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {3});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // search the path from edge1 to edge3 where all the edges are
    // targets. Path {1, 2, 3} is the shortest
    auto path = PathSearch(roadNetwork, 1, {3}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{2, 3}));

    // search the path from edge1 to edge3 where all the edges are
    // targets but edge2 is already visited. Path {1, 4, 5, 6, 3} has
    // less overhead edges
    path = PathSearch(roadNetwork, 1, {3}, {2}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{4, 5, 6, 3}));

    graphCreator
        .setAllEdgesIsTarget(true)
        .setEdgeIsTarget(2, false);
    roadNetwork = graphCreator.createGraph();

    // search the path from edge1 to edge3 where all the edges are
    // targets except edge2. Path {1, 4, 5, 6, 3} has
    // less overhead edges
    path = PathSearch(roadNetwork, 1, {3}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{4, 5, 6, 3}));

    graphCreator.setAllEdgesIsTarget(false);
    roadNetwork = graphCreator.createGraph();

    // search the path from edge1 to edge3 where all the edges are not
    // targets. Path {1, 2, 3} is the shortest
    path = PathSearch(roadNetwork, 1, {3}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{2, 3}));
}

TEST(path_search, test_search_two_finish_edges)
{
    //  graph:
    //     --5->
    //     ^    |
    //     |    6
    //     4    |
    //     |    V
    // --1->--2->--3->

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 4})
        .addEdge(2, {1, 0}, {2, 0}, {3})
        .addEdge(3, {2, 0}, {3, 0}, {})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {3});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // search the path from edge1 to edge3 or to edge6 where all the
    // edges are targets. Path to edge3 is shorter.
    auto path = PathSearch(roadNetwork, 1, {3, 6}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{2, 3}));

    // search the path from edge1 to edge3 or to edge4 where all the
    // edges are targets, but edge2 is already visited.
    // Path to edge5 has less overhead edges.
    path = PathSearch(roadNetwork, 1, {3, 5}, {2}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{4, 5}));
}

TEST(path_search, test_search_loop)
{
    //  graph:
    //     --8->
    //     ^    |
    //     |    9
    //     7    |
    //     |    V
    //     --5->
    //     ^    |
    //     |    6
    //     4    |
    //     |    V
    //      <-2--

    GraphCreator graphCreator;
    graphCreator
        .addEdge(2, {1, 0}, {2, 0}, {4})
        .addEdge(4, {1, 0}, {1, 1}, {5, 7})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {2})
        .addEdge(7, {1, 1}, {1, 2}, {8})
        .addEdge(8, {1, 2}, {2, 2}, {9})
        .addEdge(9, {2, 2}, {2, 1}, {6});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // search the path from edge2 to itself where all the edges are
    // targets. Path {2, 4, 5, 6, 2} is the shortest
    auto path = PathSearch(roadNetwork, 2, {2}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{4, 5, 6, 2}));

    // search the path from edge2 to itself where all the edges are
    // targets but edge6 is already visited. Path {2, 4, 5, 6, 2} is the shortest
    path = PathSearch(roadNetwork, 2, {2}, {6}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{4, 5, 6, 2}));

    // search the path from edge2 to itself where all the edges are
    // targets but edge5 and edge6 are already visited
    // Path {2, 4, 7, 8, 9, 6, 2} has less overhead edges
    path = PathSearch(roadNetwork, 2, {2}, {5, 6}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{4, 7, 8, 9, 6, 2}));

    // search the path from edge2 to itself where all the edges are
    // targets but edge5 and edge8 are already visited
    // in this case path {2 4 7 8 9 6 2} has bigger distance, but it has
    // better coverage, so it is better.
    path = PathSearch(roadNetwork, 2, {2}, {5, 8}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{4, 7, 8, 9, 6, 2}));

    graphCreator
        .setAllEdgesIsTarget(true)
        .setEdgeIsTarget(5, false);
    roadNetwork = graphCreator.createGraph();

    // search the path from edge2 to itself where all the edges are
    // targets except edge5
    // Path {2, 4, 7, 8, 9, 6, 2} has less overhead edges
    path = PathSearch(roadNetwork, 2, {2}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{4, 7, 8, 9, 6, 2}));
}

TEST(path_search, test_search_right_and_left_turns_penalty)
{
    //  graph:
    //          ^
    //          9
    //          |
    //          |
    //      --5->
    //          |
    //          6
    //          |
    //          V
    GraphCreator graphCreator;
    graphCreator
        .addEdge(5, {1, 1}, {2, 1}, {6, 9})
        .addEdge(6, {2, 1}, {2, 0.0}, {}, false)
        .addEdge(9, {2, 1}, {2, 1.99}, {}, false);
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // Search the path from edge5 to edge6 or edge9 where only edge5 is target.
    // Edge6 and edge9 have almost the same length(edge9 is little bit
    // smaller), but right turns are better then left turns,
    // so path {5, 6} is better
    auto path = PathSearch(roadNetwork, 5, {6, 9}, {}).getResult();
    EXPECT_TRUE((path == std::vector<EdgeId>{6}));
}

} // namespace maps::mrc::gen_targets::tests
