#include "graph_creator.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/route_generator.h>

namespace maps::mrc::gen_targets::tests {

TEST(route_generator, test_one_loop)
{
    //  graph:
    //     --5->
    //     ^    |
    //     |    6
    //     4    |
    //     |    V
    //--1-> <-2-- --3->

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {4}, false)
        .addEdge(2, {2, 0}, {1, 0}, {4})
        .addEdge(3, {2, 0}, {3, 0}, {}, false)
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6})
        .addEdge(6, {2, 1}, {2, 0}, {2, 3});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // generate loop with edge2 where all the edges are targets
    LoopsPath path = RouteGenerator(roadNetwork, 2).getResult();
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1}, {6, 1}, {2, 1}}));

    graphCreator
        .setEdgeIsTarget(4, false);
    roadNetwork = graphCreator.createGraph();

    // generate loop with edge2 where all the edges are targets except
    // edge4
    path = RouteGenerator(roadNetwork, 2).getResult();
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1}, {6, 1}, {2, 1}}));
}

TEST(route_generator, test_2_loops)
{
    //  graph:
    //          --8->
    //          ^    |
    //          |    9
    //          7    |
    //          |    V
    //     --5-><-10--
    //     ^    |
    //     |    6
    //     4    |
    //     |    V
    //      <-2--

    GraphCreator graphCreator;
    graphCreator
        .addEdge(2, {1, 0}, {2, 0}, {4})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6, 7})
        .addEdge(6, {2, 1}, {2, 0}, {2})
        .addEdge(7, {2, 1}, {2, 2}, {8})
        .addEdge(8, {2, 2}, {3, 2}, {9})
        .addEdge(9, {3, 2}, {3, 1}, {10})
        .addEdge(10, {3, 1}, {2, 1}, {7, 6});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // generate loops from edge2. Generator should generate loop
    //[4 5 6 2] and then generate and attach loop [7 8 9 10]
    LoopsPath path = RouteGenerator(roadNetwork, 2).getResult();
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1}, {7, 2}, {8, 2}, {9, 2}, {10, 2}, {6, 1}, {2, 1}}));

    graphCreator
        .setAllEdgesIsTarget(true)
        .setEdgeIsTarget(8, false)
        .setEdgeIsTarget(9, false)
        .setEdgeIsTarget(10, false);
    roadNetwork = graphCreator.createGraph();

    // Generate loops from edge2. Edges 8,9,10 are not targets.
    // Generator should generate loop
    // [4 5 6 2] and then generate and attach loop [7 8 9 10]
    path = RouteGenerator(roadNetwork, 2).getResult();
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1}, {7, 2}, {8, 2}, {9, 2}, {10, 2}, {6, 1}, {2, 1}}));

    graphCreator
        .setAllEdgesIsTarget(true)
        .setEdgeIsTarget(7, false);
    roadNetwork = graphCreator.createGraph();

    // Generate loops from edge2. Edges 7 is not targets.
    // Generator should generate only one loop [4 5 6 2] (because
    // there are not connected out edges from loop [4 5 6 2])
    path = RouteGenerator(roadNetwork, 2).getResult();
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1}, {6, 1}, {2, 1}}));
}

TEST(route_generator, test_2_loops_with_prohibited_turn)
{
    //  graph:
    //          --8->
    //          ^    |
    //          |    9
    //          7    |
    //          |    V
    //     --5-><-10--
    //     ^    |
    //     |    6
    //     4    |
    //     |    V
    //      <-2--
    //
    // turn 10->7 is prohibited

    GraphCreator graphCreator;
    graphCreator
        .addEdge(2, {1, 0}, {2, 0}, {4})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6, 7})
        .addEdge(6, {2, 1}, {2, 0}, {2})
        .addEdge(7, {2, 1}, {2, 2}, {8})
        .addEdge(8, {2, 2}, {3, 2}, {9})
        .addEdge(9, {3, 2}, {3, 1}, {10})
        .addEdge(10, {3, 1}, {2, 1}, {6});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // Generate loops from edge2. Maneuver 10->7 is not allowed.
    // Generator should generate loop
    // [4 5 6 2] and then generate and attach loop [7 8 9 10]
    LoopsPath path = RouteGenerator(roadNetwork, 2).getResult();
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1}, {7, 2}, {8, 2}, {9, 2}, {10, 2}, {6, 1}, {2, 1}}));
}

TEST(route_generator, test_loops_order)
{
    //  graph:
    // <-12-      --8->
    // |    ^    ^    |
    // 13   |    |    9
    // |    11   7    |
    // v    |    |    V
    // -14->--5-><-10--
    //      ^    |
    //      |    6
    //      4    |
    //      |    V
    //       <-2--

    GraphCreator graphCreator;
    graphCreator
        .addEdge(2, {1, 0}, {2, 0}, {4})
        .addEdge(4, {1, 0}, {1, 1}, {5, 11})
        .addEdge(5, {1, 1}, {2, 1}, {6, 7})
        .addEdge(6, {2, 1}, {2, 0}, {2})
        .addEdge(7, {2, 1}, {2, 2}, {8})
        .addEdge(8, {2, 2}, {3, 2}, {9})
        .addEdge(9, {3, 2}, {3, 1}, {10})
        .addEdge(10, {3, 1}, {2, 1}, {7, 6})
        .addEdge(11, {1, 1}, {1, 2}, {12})
        .addEdge(12, {1, 2}, {0, 2}, {13})
        .addEdge(13, {0, 2}, {0, 1}, {14})
        .addEdge(14, {0, 1}, {1, 1}, {11, 5});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // Generate loops from edge2
    // Generator should generate loop
    // [4 5 6 2], then  generate and attach loop [11 12 13 14], then
    // generate and attach loop [7 8 9 10]
    // loop [11 12 13 14] is attached first because maneuver 4->11 is better
    // then 5->7
    LoopsPath path = RouteGenerator(roadNetwork, 2).getResult();
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {11, 2}, {12, 2}, {13, 2}, {14, 2}, {5, 1}, {7, 3}, {8, 3}, {9, 3}, {10, 3}, {6, 1}, {2, 1}}));

    // the same test but from another start edge
    path = RouteGenerator(roadNetwork, 6).getResult();
    EXPECT_TRUE((path == LoopsPath{{2, 1}, {4, 1}, {11, 2}, {12, 2}, {13, 2}, {14, 2}, {5, 1}, {7, 3}, {8, 3}, {9, 3}, {10, 3}, {6, 1}}));

}

TEST(route_generator, test_transitive_loops_join)
{
    //  graph:
    //       <-12-
    //      |    ^
    //      13   |
    //      |    11
    //      v    |
    //      -14-->
    //            --8->-15->
    //           ^    |
    //           |    9
    //           7    |
    //           |    V
    //      --5-><-10--
    //      ^    |
    //      |    6
    //      4    |
    //      |    V
    //       <-2--

    GraphCreator graphCreator;
    graphCreator
        .addEdge(2, {1, 0}, {2, 0}, {4})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6, 7})
        .addEdge(6, {2, 1}, {2, 0}, {2})
        .addEdge(7, {2, 1}, {2, 2}, {8, 11})
        .addEdge(8, {2, 2}, {3, 2}, {9})
        .addEdge(9, {3, 2}, {3, 1}, {10})
        .addEdge(10, {3, 1}, {2, 1}, {7, 6})
        .addEdge(11, {1, 1}, {1, 2}, {12})
        .addEdge(12, {1, 2}, {0, 2}, {13})
        .addEdge(13, {0, 2}, {0, 1}, {14})
        .addEdge(14, {0, 1}, {1, 1}, {11, 8})
        .addEdge(15, {3, 2}, {4, 2}, {});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // Generate loops from edge2
    // Generator should generate loop
    // [4 5 6 2], then generate and attach loop [7 8 9 10],
    // then generate and attach loop [11 12 13 14]
    LoopsPath path = RouteGenerator(roadNetwork, 2).getResult();
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1}, {7, 2}, {11, 3}, {12, 3}, {13, 3}, {14, 3}, {8, 2}, {9, 2}, {10, 2}, {6, 1}, {2, 1}}));
}


TEST(route_generator, test_T_crossroad)
{
    //  graph:
    //      ^ |
    //      | |
    //      1 2
    //      | v
    // --6--> ----4---->
    // <--5-- <----3----

    // uturns on the crossroad are not allowed

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {1, 0}, {1, 1}, {2})
        .addEdge(2, {1, 1}, {1, 0}, {5, 4})
        .addEdge(3, {3, 0}, {1, 0}, {1, 5})
        .addEdge(4, {1, 0}, {3, 0}, {3})
        .addEdge(5, {1, 0}, {0, 0}, {6})
        .addEdge(6, {0, 0}, {1, 0}, {1, 4});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // Generate loops from edge1
    // Generator should generate loop
    // [2 5 6 1], then generate and attach loop [4 3]
    // loop [2 5 6 1] is attached first  because loop [2 4 3 1] is longer
    LoopsPath path = RouteGenerator(roadNetwork, 1).getResult();
    EXPECT_TRUE((path == LoopsPath{{2, 1}, {5, 1}, {6, 1}, {4, 2}, {3, 2}, {1, 1}}));
}

TEST(route_generator, test_X_crossroad)
{
    //  graph:
    //      ^ |
    //      | |
    //      1 2
    //      | v
    // --6--> <--3--
    // <--5-- --4-->
    //      ^ |
    //      | |
    //      7 8
    //      | v

    // uturns on the crossroad are not allowed

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {1, 0}, {1, 1}, {2})
        .addEdge(2, {1, 1}, {1, 0}, {5, 4, 8})
        .addEdge(3, {2, 0}, {1, 0}, {1, 5, 8})
        .addEdge(4, {1, 0}, {2, 0}, {3})
        .addEdge(5, {1, 0}, {0, 0}, {6})
        .addEdge(6, {0, 0}, {1, 0}, {1, 4, 8})
        .addEdge(7, {1, -1}, {1, 0}, {1, 4, 5})
        .addEdge(8, {1, 0}, {1, -1}, {7});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    // Generate loops from edge1
    // Generator should generate loop
    // [2 8 7 1], then generate and attach loop [4 3 5 6] using
    // a right turn
    LoopsPath path = RouteGenerator(roadNetwork, 1).getResult();
    EXPECT_TRUE((path == LoopsPath{{2, 1}, {5, 2}, {6, 2}, {4, 3}, {3, 3}, {8, 1}, {7, 1}, {1, 1}}
                 || path == LoopsPath{{2, 1}, {5, 2}, {6, 2}, {4, 2}, {3, 2}, {8, 1}, {7, 1}, {1, 1}}
                 || path == LoopsPath{{2, 1}, {8, 1}, {7, 1}, {4, 2}, {3, 2}, {5, 3}, {6, 3}, {1, 1}}
                 || path == LoopsPath{{2, 1}, {8, 1}, {7, 1}, {4, 2}, {3, 2}, {5, 2}, {6, 2}, {1, 1}}));
}

TEST(route_generator, test_delete_edges)
{
    //  graph:
    //      ^ |
    //      | |
    //      1 2
    //      | v
    // --6--> ----4---->
    // <--5-- <----3----

    // uturns on the crossroad are not allowed

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {1, 0}, {1, 1}, {2})
        .addEdge(2, {1, 1}, {1, 0}, {5, 4})
        .addEdge(3, {3, 0}, {1, 0}, {1, 5})
        .addEdge(4, {1, 0}, {3, 0}, {3})
        .addEdge(5, {1, 0}, {0, 0}, {6})
        .addEdge(6, {0, 0}, {1, 0}, {1, 4});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    LoopsPath path = RouteGenerator(roadNetwork, 1).getResult();
    EXPECT_TRUE((path == LoopsPath{{2, 1}, {5, 1}, {6, 1}, {4, 2}, {3, 2}, {1, 1}}));

    roadNetwork.deleteEdges({3, 4});
    path = RouteGenerator(roadNetwork, 1).getResult();
    EXPECT_TRUE((path == LoopsPath{{2, 1}, {5, 1}, {6, 1}, {1, 1}}));
}

} // namespace maps::mrc::gen_targets::tests
