#include "graph_creator.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/route_generator.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/unrecommended_maneuvers.h>

namespace maps::mrc::gen_targets::tests {

TEST(unrecommended_maneuvers_tests, test_unrecommended_turns_2_connected_edges)
{
    //  graph:
    //
    //--1->==2=>==3=>--4->

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2}, false)
        .addEdge(2, {1, 0}, {2, 0}, {3})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {4, 0}, {}, false);
    ExtendedRoadNetwork roadNetwork(graphCreator.createGraph().getEdges());
    auto unrecommendedManeuvers
        = UnrecommendedManeuversCalculator(roadNetwork).getUnrecommendedManeuvers();
    EXPECT_EQ(unrecommendedManeuvers.size(), 0u);
}

TEST(unrecommended_maneuvers_tests, test_unrecommended_turns_T_crossroad)
{
    //  graph:
    //
    //          ^
    //          |
    //          3
    //          |
    //--5-->--1->--2->--4->

    // only two target edges, so no unrecommended turns

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 3})
        .addEdge(2, {1, 0}, {2, 0}, {4})
        .addEdge(3, {1, 0}, {1, 1}, {}, false)
        .addEdge(4, {2, 0}, {3, 0}, {}, false)
        .addEdge(5, {-1, 0}, {0, 0}, {1}, false);
    ExtendedRoadNetwork roadNetwork(graphCreator.createGraph().getEdges());

    auto unrecommendedManeuvers
        = UnrecommendedManeuversCalculator(roadNetwork).getUnrecommendedManeuvers();
    EXPECT_EQ(unrecommendedManeuvers.size(), 0u);
}

TEST(unrecommended_maneuvers_tests, test_unrecommended_turns_simple_2_way_crossroad)
{
    //  graph:
    //         ^
    //         7
    //         ^
    //         |
    //         3
    //         |
    //-5->--1-> --2->-6>
    //         ^
    //         |
    //         4
    //         |
    //         ^
    //         8

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 3})
        .addEdge(2, {1, 0}, {2, 0}, {6})
        .addEdge(3, {1, 0}, {1, 1}, {7})
        .addEdge(4, {1, -1}, {1, 0}, {2, 3})
        .addEdge(5, {-1, 0}, {0, 0}, {1}, false)
        .addEdge(6, {2, 0}, {3, 0}, {}, false)
        .addEdge(7, {1, 1}, {1, 2}, {}, false)
        .addEdge(8, {1, -2}, {1, -1}, {4}, false);
    ExtendedRoadNetwork roadNetwork(graphCreator.createGraph().getEdges());

    auto unrecommendedManeuvers
        = UnrecommendedManeuversCalculator(roadNetwork).getUnrecommendedManeuvers();
    EXPECT_EQ(unrecommendedManeuvers.size(), 0u);
}

TEST(unrecommended_maneuvers_tests, test_unrecommended_turns_2_way_crossroad_with_prohibited_turn)
{
    //  graph:
    //         ^
    //         7
    //         ^
    //         |
    //         3
    //         |
    //-5->--1-> --2->-6>
    //         ^
    //         |
    //         4
    //         |
    //         ^
    //         8

    // Edges 1,2,3,4 are targets.
    // Turn 4->3 is prohibited.
    // edge2 is the only way from edge4
    // so maneuver 1->2 is not recommended

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 3})
        .addEdge(2, {1, 0}, {2, 0}, {})
        .addEdge(3, {1, 0}, {1, 1}, {})
        .addEdge(4, {1, -1}, {1, 0}, {2})
        .addEdge(5, {-1, 0}, {0, 0}, {1}, false)
        .addEdge(6, {2, 0}, {3, 0}, {}, false)
        .addEdge(7, {1, 1}, {1, 2}, {}, false)
        .addEdge(8, {1, -2}, {1, -1}, {4}, false);
    ExtendedRoadNetwork roadNetwork(graphCreator.createGraph().getEdges());

    auto unrecommendedManeuvers
            = UnrecommendedManeuversCalculator(roadNetwork).getUnrecommendedManeuvers();
    EXPECT_EQ(unrecommendedManeuvers.size(), 1u);
    EXPECT_TRUE(unrecommendedManeuvers.count(1));
    EXPECT_THAT(unrecommendedManeuvers[1], testing::UnorderedElementsAre(2));
}

TEST(unrecommended_maneuvers_tests, test_unrecommended_turns_3_way_crossroad_with_prohibited_turns)
{
    //  graph:
    //         ^
    //         7
    //         ^
    //         |
    //         3
    //         |
    //-5->--1-> --2->-6>
    //11->--9-> -10->12>
    //         ^
    //         |
    //         4
    //         |
    //         ^
    //         8

    // Edges 1,2,3,4,9,10 are targets.
    // Turns 1->3, 9->2, 4->10, 4->2 are prohibited.
    // edge3 is the only way from edge4, that means that edge 10 is
    // the only way from edge 9, that means that edge 2 is the only
    // way from edge 1.
    // So maneuvers 1->10 and 9-3 are unrecommended

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 10})
        .addEdge(9, {0, 0}, {1, 0}, {10, 3})
        .addEdge(2, {1, 0}, {2, 0}, {})
        .addEdge(10, {1, 0}, {2, 0}, {12})
        .addEdge(3, {1, 0}, {1, 1}, {})
        .addEdge(4, {1, -1}, {1, 0}, {3})
        .addEdge(5, {-1, 0}, {0, 0}, {1}, false)
        .addEdge(6, {2, 0}, {3, 0}, {}, false)
        .addEdge(7, {1, 1}, {1, 2}, {}, false)
        .addEdge(8, {1, -2}, {1, -1}, {4}, false)
        .addEdge(11, {-1, 0}, {0, 0}, {9}, false)
        .addEdge(12, {2, 0}, {3, 0}, {}, false);
    ExtendedRoadNetwork roadNetwork(graphCreator.createGraph().getEdges());

    auto unrecommendedManeuvers
            = UnrecommendedManeuversCalculator(roadNetwork).getUnrecommendedManeuvers();
    EXPECT_EQ(unrecommendedManeuvers.size(), 2u);
    EXPECT_TRUE(unrecommendedManeuvers.count(1));
    EXPECT_THAT(unrecommendedManeuvers[1], testing::UnorderedElementsAre(10));
    EXPECT_TRUE(unrecommendedManeuvers.count(9));
    EXPECT_THAT(unrecommendedManeuvers[9], testing::UnorderedElementsAre(3));
}

TEST(unrecommended_maneuvers_tests, test_unrecommended_in_road_network)
{
    //  graph:
    //
    //--1->==2=>==3=>--4->

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2}, false)
        .addEdge(2, {1, 0}, {2, 0}, {3})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {4, 0}, {}, false);
    RoadNetworkData roadNetwork = graphCreator.createGraph();
    ExtendedRoadNetwork extendedRoadNetwork(graphCreator.createGraph().getEdges());
    const RoadNetworkData& pExtendedRoadNetwork = extendedRoadNetwork;

    EXPECT_EQ(roadNetwork.getUnrecommendedManeuversPenalty(2, 3), 0.0);
    EXPECT_EQ(roadNetwork.getUnrecommendedManeuversPenalty(3, 4), 0.0);
    EXPECT_EQ(pExtendedRoadNetwork.getUnrecommendedManeuversPenalty(2, 3), 0.0);
    EXPECT_EQ(pExtendedRoadNetwork.getUnrecommendedManeuversPenalty(3, 4), 0.0);

    // add one unrecommended maneuver 2->3
    extendedRoadNetwork.setUnrecommendedManeuvers({{2, {3}}});

    EXPECT_EQ(pExtendedRoadNetwork.getUnrecommendedManeuversPenalty(2, 3),
              UNRECOMMENDED_MANEUVER_PENALTY);
    EXPECT_EQ(pExtendedRoadNetwork.getUnrecommendedManeuversPenalty(3, 4), 0.0);
}

TEST(unrecommended_maneuvers_tests, test_unrecommended_maneuvers_in_loop_connections)
{
    //  graph:
    //            --8->
    //            ^    |
    //            |    9  loop2
    //            7    |
    //loop1       |    V
    //     --5->  <-10--
    //     ^    | --11->
    //     |    6 ^    |
    //     4    | |   14
    //     |    V 12   |  loop3
    //      <-2-- |    V
    //            <-13--

    // loops are connected only in one node

    GraphCreator graphCreator;
    graphCreator
        .addEdge(2, {1, 0}, {2, 0}, {4})
        .addEdge(4, {1, 0}, {1, 1}, {5})
        .addEdge(5, {1, 1}, {2, 1}, {6, 7, 11})
        .addEdge(6, {2, 1}, {2, 0}, {2})
        .addEdge(7, {2, 1}, {2, 2}, {8})
        .addEdge(8, {2, 2}, {3, 2}, {9})
        .addEdge(9, {3, 2}, {3, 1}, {10})
        .addEdge(10, {3, 1}, {2, 1}, {6, 7, 11})
        .addEdge(11, {2, 1}, {3, 1}, {14})
        .addEdge(12, {2, 0}, {2, 1}, {6, 7, 11})
        .addEdge(13, {3, 0}, {2, 0}, {12})
        .addEdge(14, {3, 1}, {3, 0}, {13});

    // add additional time to loop2 and loop3 to force generator not
    // to create one big loop.
    graphCreator.edges_[8].time = 10000;
    graphCreator.edges_[13].time = 10000;
    ExtendedRoadNetwork roadNetwork(graphCreator.createGraph().getEdges());

    // generate loops from edge2.
    LoopsPath path = RouteGenerator(roadNetwork, 2).getResult();
    // expect loops connections with the best maneuvers
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1},  // loop1
                                   {11, 2}, {14, 2}, {13, 2}, {12, 2}, // loop3
                                   {7, 3}, {8, 3}, {9, 3}, {10, 3}, // loop2
                                   {6, 1}, {2, 1}})); // loop1

    roadNetwork.setUnrecommendedManeuvers({{5, {11}}});

    path = RouteGenerator(roadNetwork, 2).getResult();
    // expect loops connections without unrecommended maneuvers
    EXPECT_TRUE((path == LoopsPath{{4, 1}, {5, 1},  // loop 1
                                   {7, 2}, {8, 2}, {9, 2}, {10, 2}, // loop2
                                   {11, 3}, {14, 3}, {13, 3}, {12, 3}, // loop3
                                   {6, 1}, {2, 1}})); // loop1
}

} // namespace maps::mrc::gen_targets::tests
