#include "graph_creator.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/districts.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/static_graph_loader.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/tasks_gen/lib/tasks_generator.h>

namespace maps::mrc::gen_targets::tests {

bool pathsAreEqual(const Path& getPath, std::vector<EdgeId> expPath)
{
    if (getPath.size() != expPath.size()) {
        return false;
    }

    for (size_t shift = 0; shift < getPath.size(); shift++) {
        bool equal = true;
        for (size_t i = 0; i < getPath.size(); i++) {
            if (getPath[i].edgeId != expPath[(i + shift) % getPath.size()]) {
                equal = false;
            }
        }
        if (equal) {
            return true;
        }
    }
    return false;
}

bool tasksAreEqual(const std::vector<std::pair<District, Path>>& getTasks,
                   std::vector<std::vector<EdgeId>> expTasks) {
    if (getTasks.size() != expTasks.size()) {
        return false;
    }

    for(const auto& expTask : expTasks) {
        bool foundEqual = false;
        for(const auto& getTask : getTasks) {
            if (pathsAreEqual(getTask.second, expTask)) {
                foundEqual = true;
            }
        }
        if (!foundEqual) {
            return false;
        }
    }

    return true;
}

TEST(utils_and_coomon, test_edges_inside_districts)
{
    //  graph:
    // ----------------------
    // |                    |
    // | <-6-- <-5-- <-4--  |
    // | --1-> --2-> --3->  | --7-->
    // |                    |
    // |  polygon           |
    // ----------------------

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 6})
        .addEdge(2, {1, 0}, {2, 0}, {3, 5})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {2, 0}, {3, 5})
        .addEdge(5, {2, 0}, {1, 0}, {2, 6})
        .addEdge(6, {1, 0}, {0, 0}, {1})
        .addEdge(7, {20, 0}, {21, 0}, {});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto polygon = createPolygon({{-1, 1}, {4, 1}, {4, -1}, {-1, -1}});

    auto edgesInside = roadNetwork.getEdgesWithinPolygon(polygon, 1, 10, {});

    // should return all the edges inside the polygon
    EXPECT_TRUE((std::set<EdgeId>(edgesInside.begin(), edgesInside.end())
                 == std::set<EdgeId>{1, 2, 3, 4, 5, 6}));

    graphCreator.edges_[1].fc = 0;
    graphCreator.edges_[2].fc = 20;
    roadNetwork = graphCreator.createGraph();

    // edges 1 and 2 has bad fc, edge 3 is prohibited
    // only edges 4, 5, 6 should be returned
    edgesInside = roadNetwork.getEdgesWithinPolygon(polygon, 1, 10, {3});
    EXPECT_TRUE((std::set<EdgeId>(edgesInside.begin(), edgesInside.end())
                 == std::set<EdgeId>{4, 5, 6}));
}

TEST(utils_and_coomon, test_edges_mostly_inside_districts)
{
    //  graph:
    //   -------------
    //   |           |
    //<-6-- <-5-- <-4--
    //--1-> --2-> --3->
    //   |           |
    //   |  polygon  |
    //   -------------

    // edges 1, 6 are mostly outside the polygon
    // edges 4, 3 are mostly inside the polygon
    // the generator should consider it and return only edges that are
    // inside the polygon or mostly inside the polygon

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 6})
        .addEdge(2, {1, 0}, {2, 0}, {3, 5})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {2, 0}, {3, 5})
        .addEdge(5, {2, 0}, {1, 0}, {2, 6})
        .addEdge(6, {1, 0}, {0, 0}, {1});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto polygon = createPolygon({{0.7, 1}, {2.7, 1}, {2.7, -1}, {0.7, -1}});

    auto edgesInside = roadNetwork.getEdgesWithinPolygon(polygon, 0, 10, {});

    EXPECT_TRUE((std::set<EdgeId>(edgesInside.begin(), edgesInside.end())
                 == std::set<EdgeId>{2, 3, 4, 5}));
}

TEST(utils_and_coomon, test_generate_tasks_inside_polygon)
{
    //  graph:
    //   -------------
    //   |           |
    //<-6-- <-5-- <-4--
    //--1-> --2-> --3->
    //   |           |
    //   |  polygon  |
    //   -------------

    // all U turns are allowed
    // only edges 2 3 4 5 are inside the polygon

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 6})
        .addEdge(2, {1, 0}, {2, 0}, {3, 5})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {2, 0}, {3, 5})
        .addEdge(5, {2, 0}, {1, 0}, {2, 6})
        .addEdge(6, {1, 0}, {0, 0}, {1})
        .setAllEdgesIsTarget(false);
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto polygon = createPolygon({{0.7, 1}, {2.7, 1}, {2.7, -1}, {0.7, -1}});
    District district;
    district.area = polygon;

    // generator should return one task with all the edges inside the polygon
    std::unordered_set<EdgeId> prohibitedEdges;
    auto tasks = generateTasksWithinDistricts(roadNetwork, {district},
                                              10, 4000,
                                              prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {{2, 3, 4, 5}}));

    // generator should return the same, but divided into 2 tasks
    // because each loop > 1500m
    prohibitedEdges = {};
    tasks = generateTasksWithinDistricts(roadNetwork, {district},
                                         10, 1500,
                                         prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {{2, 5}, {3, 4}}));

    // excluding edges 3, 4 from possible targets
    prohibitedEdges = {3, 4};
    tasks = generateTasksWithinDistricts(roadNetwork, {district},
                                         10, 4000,
                                         prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {{2, 5}}));

    // excluding edge 4 from possible targets
    // generator should include edge 4 to maintain the task continuity
    prohibitedEdges = {4};
    tasks = generateTasksWithinDistricts(roadNetwork, {district},
                                         10, 4000,
                                         prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {{2, 3, 4, 5}}));

    // excluding all the edges inside the polygon from possible targets
    prohibitedEdges = {3, 4, 2, 5};
    tasks = generateTasksWithinDistricts(roadNetwork, {district},
                                         10, 1500,
                                         prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {}));


    // excluding edges 2, 5 from possible targets by setting bad edge fc
    graphCreator.edges_[2].fc = 11;
    graphCreator.edges_[5].fc = 20;
    roadNetwork = graphCreator.createGraph();
    prohibitedEdges = {};
    tasks = generateTasksWithinDistricts(roadNetwork, {district},
                                         10, 1500,
                                         prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {{3, 4}}));

}

TEST(utils_and_coomon, test_generate_unconnected_tasks)
{
    //  graph:
    // ----------------------
    // |                    |
    // | <-6--<-5--<-4--  |
    // | --1->--2->--3->  |
    // |                    |
    // |  polygon           |
    // ----------------------

    // edges 2 and 5 has non-target fc

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2, 6})
        .addEdge(2, {1, 0}, {2, 0}, {3})
        .addEdge(6, {1, 0}, {0, 0}, {1})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {2, 0}, {3, 5})
        .addEdge(5, {2, 0}, {1, 0}, {6});
    graphCreator.edges_[2].fc = 20;
    graphCreator.edges_[5].fc = 20;
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto polygon = createPolygon({{-1, 1}, {4, 1}, {4, -1}, {-1, -1}});
    District district;
    district.area = polygon;
    std::unordered_set<EdgeId> prohibitedEdges = {};

    // There is one strongly connected components of edges and
    // 2 weakly connected components of targets.
    // Generator should create a separate task for each
    // weakly connected component of targets
    auto tasks = generateTasksWithinDistricts(roadNetwork, {district},
                                              10, 4000,
                                              prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {{1, 6}, {4, 3}}));
}

TEST(utils_and_coomon, test_exclude_barrier_area)
{
    //  graph:
    // ----------------------
    // |                    |
    // | <-6--<-5--<--4--  |
    // | --1->--2->|--3->  |
    // |                    |
    // |  polygon           |
    // ----------------------

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {2})
        .addEdge(2, {1, 0}, {2, 0}, {5})
        .addEdge(5, {2, 0}, {1, 0}, {6})
        .addEdge(6, {1, 0}, {0, 0}, {1})
        .addEdge(3, {2, 0}, {3, 0}, {4})
        .addEdge(4, {3, 0}, {2, 0}, {3, 5});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto polygon = createPolygon({{-1, 1}, {4, 1}, {4, -1}, {-1, -1}});
    District district;
    district.area = polygon;
    std::unordered_set<EdgeId> prohibitedEdges = {};

    // There are 2 strongly connected components of edges.
    // Generator should create a task only for the largest component
    auto tasks = generateTasksWithinDistricts(roadNetwork, {district},
                                              10, 4000,
                                              prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {{1, 2, 5, 6}}));
}

TEST(utils_and_coomon, test_generate_tasks_in_two_districts)
{
    //  graph:
    // -----------------------
    // |         |           |
    // |    <-6--|<-4--      |
    // |    --1->|--3->      |
    // |         |           |
    // |polygon1 | polygon2  |
    // -----------------------

    GraphCreator graphCreator;
    graphCreator
        .addEdge(1, {0, 0}, {1, 0}, {3, 6})
        .addEdge(3, {1, 0}, {2, 0}, {4})
        .addEdge(4, {2, 0}, {1, 0}, {3, 6})
        .addEdge(6, {1, 0}, {0, 0}, {1});
    RoadNetworkData roadNetwork = graphCreator.createGraph();

    auto polygon1 = createPolygon({{-1, 1}, {1, 1}, {1, -1}, {-1, -1}});
    auto polygon2 = createPolygon({{1, 1}, {4, 1}, {4, -1}, {1, -1}});
    District district1;
    district1.area = polygon1;
    District district2;
    district2.area = polygon2;
    std::unordered_set<EdgeId> prohibitedEdges = {};

    // There is astrongly connected graph and 2 districts with targets.
    // Generator should create a separate task for each district.
    auto tasks = generateTasksWithinDistricts(roadNetwork, {district1, district2},
                                              10, 4000,
                                              prohibitedEdges);
    EXPECT_TRUE(tasksAreEqual(tasks, {{1, 6}, {4, 3}}));
}

} // namespace maps::mrc::gen_targets::tests
