#pragma once

#include <maps/libs/chrono/include/days.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>

#include <cstdint>
#include <map>
#include <optional>
#include <sstream>
#include <vector>

#include <boost/container/flat_set.hpp>

namespace maps::mrc::taxi_stat {

/**
 * @see
 * https://yt.yandex-team.ru/hahn/navigation?offsetMode=key&navmode=schema&path=//home/taxi/production/replica/postgres/signal_device_api_meta_db/events
 */
using DeviceId = int64_t;
using DeviceIdSet = boost::container::flat_set<DeviceId>;
using EventId = int64_t;

using GeoId = int32_t;
using GeoIds = std::vector<GeoId>;

struct DateRegion {
    chrono::TimePoint date;
    GeoId geoId;

    friend bool operator<(const DateRegion& lhs, const DateRegion& rhs)
    {
        return std::tie(lhs.date, lhs.geoId) < std::tie(rhs.date, rhs.geoId);
    }

    friend bool operator==(const DateRegion& lhs, const DateRegion& rhs)
    {
        return std::tie(lhs.date, lhs.geoId) == std::tie(rhs.date, rhs.geoId);
    }
};

struct Aggregate {
    size_t bytesNumber;
    EventId maxEventId;
    DeviceIdSet deviceIdSet;

    friend bool operator==(const Aggregate&, const Aggregate&) = default;
};

using Log = std::map<DateRegion, Aggregate>;

inline std::optional<EventId> maxEventId(const Log& log)
{
    if (log.empty()) {
        return std::nullopt;
    }
    return std::max_element(log.begin(),
                            log.end(),
                            [](const auto& lhs, const auto& rhs) {
                                return lhs.second.maxEventId <
                                       rhs.second.maxEventId;
                            })
        ->second.maxEventId;
}

struct BanKey {
    chrono::TimePoint date;
    db::DisqType disqType;

    friend bool operator<(const BanKey& lhs, const BanKey& rhs)
    {
        return std::tie(lhs.date, lhs.disqType) <
               std::tie(rhs.date, rhs.disqType);
    }

    friend bool operator==(const BanKey& lhs, const BanKey& rhs)
    {
        return std::tie(lhs.date, lhs.disqType) ==
               std::tie(rhs.date, rhs.disqType);
    }
};

using Bans = std::map<BanKey, size_t>;

using Activity = std::map<DateRegion, size_t>;

inline chrono::TimePoint epochToDay(std::chrono::seconds::rep sec)
{
    return std::chrono::floor<chrono::Days>(chrono::TimePoint{} +
                                            std::chrono::seconds{sec});
}

inline std::chrono::seconds::rep toEpoch(chrono::TimePoint date)
{
    return std::chrono::floor<std::chrono::seconds>(date.time_since_epoch())
        .count();
}

inline std::string concat(const auto&... args)
{
    auto os = std::ostringstream{};
    ((os << args), ...);
    return os.str();
}

}  // namespace maps::mrc::taxi_stat
