#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/taxi_stat/lib/db.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/taxi_stat/lib/geobase.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/taxi_stat/lib/yt.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>

namespace taxi_stat = maps::mrc::taxi_stat;

int main(int argc, char* argv[])
try {
    NYT::Initialize(argc, argv);
    auto parser = maps::cmdline::Parser{};
    auto syslog = parser.string("syslog-tag")
                      .help("redirect log output to syslog with given tag");
    auto configPath = parser.string("config").help("path to configuration");
    auto secretVersion = parser.string("secret").help(
        "version for secrets from yav.yandex-team.ru");
    auto geodataPath = parser.string("geodata-path")
                           .defaultValue("/var/cache/geobase/geodata6.bin")
                           .help("path to coverage for geoid");
    auto geodataPatchYtDir =
        parser.string("geodata-patch-yt-dir")
            .defaultValue("//home/maps/core/nmaps/analytics/geo-data")
            .help("YT dir with additional geobase regions");
    auto dryRun = parser.flag("dry-run").help("do not publish data");
    parser.parse(argc, argv);

    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }
    auto config =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);
    auto postgres = config.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                                          maps::mrc::common::LONG_READ_POOL_ID);
    auto mutex = maps::pgp3utils::PgAdvisoryXactMutex{
        postgres.pool(),
        static_cast<int64_t>(maps::mrc::common::LockId::TaxiStat)};
    if (!dryRun && !mutex.try_lock()) {
        INFO() << "another process is ongoing";
        return EXIT_SUCCESS;
    }
    INFO() << "updating taxi statistics, dryRun=" << dryRun;

    auto now = maps::chrono::TimePoint::clock::now();
    auto ytClient = config.externals().yt().makeClient();
    auto geoEvaluatorPtr = taxi_stat::makeGeobaseEvaluator(
        ytClient, geodataPath, geodataPatchYtDir);
    auto ytTxn = ytClient->StartTransaction();

    auto eventsTable = config.taxiImportConfig().ytEventsPath() + "events";
    auto grepTable = config.externals().yt().path() + "/taxi_grep";
    auto logTable = config.externals().yt().path() + "/taxi_log";
    auto log = taxi_stat::loadLog(*ytTxn, logTable);
    taxi_stat::grepEvents(
        *ytTxn,
        eventsTable,
        grepTable,
        maxEventId(log).value_or(taxi_stat::PREVIOUS_OF_FIRST_EVENT_ID) + 1);
    taxi_stat::loadEvents(*ytTxn, *geoEvaluatorPtr, grepTable, log);
    taxi_stat::saveLog(*ytTxn, log, logTable);

    auto bansTable = config.externals().yt().path() + "/taxi_bans";
    auto bans = taxi_stat::loadBans(postgres.pool(), now);
    taxi_stat::saveBans(*ytTxn, bans, bansTable);

    auto tileRequestsTable =
        config.externals().yt().path() + "/taxi_tile_requests";
    auto activityTable = config.externals().yt().path() + "/taxi_activity";
    auto activity = taxi_stat::loadActivity(*ytTxn, activityTable);
    taxi_stat::grepTileRequestsFromDir(
        *ytTxn,
        "//home/logfeller/logs/maps-log/1d",
        tileRequestsTable,
        activity.empty() ? std::nullopt
                         : std::optional{activity.rbegin()->first.date});
    taxi_stat::loadTileRequests(
        *ytTxn, *geoEvaluatorPtr, tileRequestsTable, activity);
    taxi_stat::saveActivity(*ytTxn, activity, activityTable);

    if (!dryRun) {
        ytTxn->Commit();
        INFO() << "updating metadata";
        taxi_stat::updateLastRunTime(postgres.pool(), now);
    }

    INFO() << "done";
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
