#include "../include/cancel_tasks.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_type_info_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/toloka_task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/toloka_task_suite_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/for_each_batch.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/collection.h>

#include <maps/libs/sql_chemistry/include/exists.h>

#include <maps/libs/log8/include/log8.h>

namespace maps::mrc::toloka {

namespace {

db::toloka::TolokaTasks loadTolokaTasksToCancel(
    pqxx::transaction_base& txn, db::toloka::Platform platform)
{
    return db::toloka::TolokaTaskGateway{txn}.load(
        db::toloka::table::TolokaTask::platform == platform &&
        db::toloka::table::TolokaTask::taskId ==
            db::toloka::table::Task::id &&
        db::toloka::table::Task::status == db::toloka::TaskStatus::Cancelling);
}

db::toloka::TolokaTaskSuites
loadTolokaSuitesToCancel(pqxx::transaction_base& txn, const db::TIds& suiteIds)
{
    db::toloka::TolokaTaskSuites result;
    common::forEachBatch(
        suiteIds,
        1000,
        [&](auto begin, auto end)
        {
            db::TIds suiteIds{begin, end};

            auto tolokaTasksInProgress =
                db::toloka::TolokaTaskGateway{txn}.load(
                    db::toloka::table::TolokaTask::taskSuiteId.in(suiteIds) &&
                        db::toloka::table::TolokaTask::taskId ==
                            db::toloka::table::Task::id &&
                        db::toloka::table::Task::status.in({
                            db::toloka::TaskStatus::New,
                            db::toloka::TaskStatus::InProgress
                        }) &&
                        db::toloka::table::Task::knownSolutions.isNull()
                );
            db::TIdSet suiteIdsInProgress;
            for (const auto& tolokaTask : tolokaTasksInProgress) {
                suiteIdsInProgress.insert(tolokaTask.taskSuiteId());
            }

            suiteIds.erase(
                std::remove_if(
                    suiteIds.begin(),
                    suiteIds.end(),
                    [&](auto id) { return suiteIdsInProgress.count(id); }
                ),
                suiteIds.end()
            );

            auto suites = db::toloka::TolokaTaskSuiteGateway{txn}.loadByIds(suiteIds);

            result.insert(result.end(),
                std::make_move_iterator(suites.begin()),
                std::make_move_iterator(suites.end()));
        });
    return result;
}

} // namespace

void cancelTasks(db::toloka::Platform platform, pgpool3::Pool& pool, toloka::io::TolokaClient& client)
{
    auto tolokaTasks = loadTolokaTasksToCancel(*pool.slaveTransaction(), platform);
    db::TIdSet suiteIdsOfCancelledTasks;
    db::TIds taskIdsOfCancelledTasks;

    INFO() << "Loaded " << tolokaTasks.size() << " to cancel";

    for (const auto& tolokaTask : tolokaTasks) {
        suiteIdsOfCancelledTasks.insert(tolokaTask.taskSuiteId());
    }

    auto suitesToCancel = loadTolokaSuitesToCancel(
        *pool.slaveTransaction(),
        {suiteIdsOfCancelledTasks.begin(), suiteIdsOfCancelledTasks.end()});
    db::TIdSet cancelledSuiteIdSet;

    INFO() << "Loaded " << suitesToCancel.size() << " suites to cancel";

    for (auto& suite : suitesToCancel) {
        client.stopAssigningTaskSuite(suite.tolokaId());
        cancelledSuiteIdSet.insert(suite.id());
    }

    for (const auto& tolokaTask : tolokaTasks) {
        if (cancelledSuiteIdSet.count(tolokaTask.taskSuiteId())) {
            taskIdsOfCancelledTasks.push_back(tolokaTask.taskId());
        }
    }

    auto txn = pool.masterWriteableTransaction();
    INFO() << "Cancelled " << taskIdsOfCancelledTasks.size() << " tasks";
    common::forEachBatch(
        taskIdsOfCancelledTasks,
        1000,
        [&](auto begin, auto end)
        {
            db::toloka::TaskGateway{*txn}.updateStatusByIds(db::toloka::TaskStatus::Cancelled, {begin, end});
        });
    txn->commit();
    INFO() << "Cancel tasks done";
}

} // namespace maps::mrc::toloka
