#include "toloka/platform.h"
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/check_completed_tasks.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/common.h>

#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/approvement.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/detection.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/image_quality.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/taxonomy.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/house_number_recognition.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/detection_pair_match.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/detection_missing_on_frame.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_type_info_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/toloka_task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/toloka_task_suite_gateway.h>

#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/mrc/toloka_client/filter.h>

#include <boost/noncopyable.hpp>
#include <boost/lexical_cast.hpp>

#include <algorithm>
#include <functional>
#include <string>
#include <unordered_map>
#include <unordered_set>
#include <vector>

namespace maps {
namespace mrc {
namespace toloka {
namespace {

template <typename TaskSuiteResult>
std::vector<std::string>
toOutputValues(const TaskSuiteResult& taskSuiteResult)
{
    std::vector<std::string> outputValues;
    outputValues.reserve(taskSuiteResult.taskResults.size());
    for (const auto& taskResult : taskSuiteResult.taskResults) {
        std::ostringstream os;
        os << taskResult.output;
        outputValues.push_back(os.str());
    }
    return outputValues;
}

using IsSuiteInProgress
    = std::function<bool(const std::string& /*tolokaSuiteId*/)>;

template <typename T>
std::unordered_map<std::string, T>
filterSuites(std::unordered_map<std::string, T>&& suites,
             const IsSuiteInProgress& isSuiteInProgress)
{
    std::unordered_map<std::string, T> result;
    for (const auto& suite : suites)
        if (isSuiteInProgress(suite.first))
            result.insert({suite.first, std::move(suite.second)});
    return result;
}

using OnSolveSuite
    = std::function<void(const std::string& /*tolokaSuiteId*/,
                         const std::vector<std::string>& /*outputValues*/)>;

/// base class for handling different types of tasks
class TolokaCompletedTasksHandler : private boost::noncopyable {
public:
    virtual ~TolokaCompletedTasksHandler() = default;
    virtual db::toloka::TaskType type() = 0;
    virtual void handlePool(io::TolokaClient& client,
                            const std::string& poolId,
                            const IsSuiteInProgress& isSuiteInProgress,
                            const OnSolveSuite& onSolveSuite) = 0;
};

template <typename TaskType>
class TolokaCompletedTasksTemplateHandler : public TolokaCompletedTasksHandler {
public:
    void handlePool(io::TolokaClient& client,
                            const std::string& poolId,
                            const IsSuiteInProgress& isSuiteInProgress,
                            const OnSolveSuite& onSolveSuite) override
    {
        auto idToTaskSuite
            = filterSuites(loadTaskSuites<TaskType>(client, poolId),
                           isSuiteInProgress);
        auto idToTaskSuiteResults
            = loadTaskSuitesResults<TaskType>(client, poolId);
        auto suitIdToResultMap = mergeTasksResults<TaskType>(idToTaskSuite, idToTaskSuiteResults);
        for (const auto& suitIdToResult : suitIdToResultMap) {
            try {
                TaskType::evaluateAssignments(client, suitIdToResult.second);
                onSolveSuite(suitIdToResult.first,
                             toOutputValues(suitIdToResult.second));
            }
            catch (const maps::Exception& e) {
                WARN() << e;
            }
        }
    }
};

class TrafficLightDetectionHandler : public TolokaCompletedTasksTemplateHandler<Detection> {
public:
    db::toloka::TaskType type() override
    {
        return db::toloka::TaskType::TrafficLightDetection;
    }
};

class ImageQualityHandler : public TolokaCompletedTasksTemplateHandler<ImageQuality> {
public:
    db::toloka::TaskType type() override
    {
        return db::toloka::TaskType::ImageQualityClassification;
    }
};

class ApprovementHandler : public TolokaCompletedTasksTemplateHandler<Approvement> {
public:
    db::toloka::TaskType type() override
    {
        return db::toloka::TaskType::Approvement;
    }
};

class TaxonomyHandler : public TolokaCompletedTasksTemplateHandler<Taxonomy> {
public:
    db::toloka::TaskType type() override
    {
        return db::toloka::TaskType::Taxonomy;
    }
};

class HouseNumberDetectionHandler : public TolokaCompletedTasksTemplateHandler<Detection> {
public:
    db::toloka::TaskType type() override
    {
        return db::toloka::TaskType::HouseNumberDetection;
    }
};

class HouseNumberRecognitionHandler : public TolokaCompletedTasksTemplateHandler<HouseNumberRecognition> {
public:
    db::toloka::TaskType type() override
    {
        return db::toloka::TaskType::HouseNumberRecognition;
    }
};

class DetectionPairMatchHandler : public TolokaCompletedTasksTemplateHandler<DetectionPairMatch> {
public:
    db::toloka::TaskType type() override
    {
        return db::toloka::TaskType::DetectionPairMatch;
    }
};

class DetectionMissingOnFrameHandler : public TolokaCompletedTasksTemplateHandler<DetectionMissingOnFrame> {
public:
    db::toloka::TaskType type() override
    {
        return db::toloka::TaskType::DetectionMissingOnFrame;
    }
};


class DbContext {
public:
    DbContext(db::toloka::Platform platform, pgpool3::Pool& pool, db::toloka::TaskType type)
        : pool_(pool)
        , platform_(platform)
        , type_(type)
        , tolokaIdToSuiteMap_{loadTolokaIdToSuiteMap()}
    {
    }

    std::string name() const
    {
        return boost::lexical_cast<std::string>(type_);
    }

    db::TIds inProgressPoolIds() const
    {
        std::unordered_set<db::TId> poolIds;
        for (const auto& tolokaIdToSuiteInfo : tolokaIdToSuiteMap_) {
            const auto& dbSuite = tolokaIdToSuiteInfo.second.dbSuite;
            if (!dbSuite.solvedAt()) {
                poolIds.insert(dbSuite.tolokaPoolId());
            }
        }
        return {poolIds.begin(), poolIds.end()};
    }

    bool isSuiteInProgress(const std::string& tolokaSuiteId)
    {
        return tolokaIdToSuiteMap_.count(tolokaSuiteId)
            && !tolokaIdToSuiteMap_.at(tolokaSuiteId).dbSuite.solvedAt();
    }

    void saveSuite(const std::string& tolokaSuiteId,
                   const std::vector<std::string>& outputValues)
    {
        INFO() << "Saving suite " << tolokaSuiteId;
        REQUIRE(tolokaIdToSuiteMap_.count(tolokaSuiteId),
                "Unknown suite: " << tolokaSuiteId);
        const auto NOW = chrono::TimePoint::clock::now();

        auto& info = tolokaIdToSuiteMap_.at(tolokaSuiteId);
        info.dbSuite.setSolvedAt(NOW);

        auto txn = pool_.masterWriteableTransaction();
        db::toloka::TolokaTaskSuiteGateway{*txn}.update(info.dbSuite);

        // info.dbTaskIds contains zeros in positions of golden tasks,
        // so we remove zeros before loading tasks from db
        db::TIds dbTaskIds = info.dbTaskIds;
        dbTaskIds.erase(std::remove_if(dbTaskIds.begin(), dbTaskIds.end(),
                [](auto id){ return !id; }), dbTaskIds.end());
        auto dbTasks = db::toloka::TaskGateway{*txn}.loadByIds(dbTaskIds);
        ASSERT(dbTasks.size() == dbTaskIds.size());

        dbTasks.erase(
            std::remove_if(dbTasks.begin(), dbTasks.end(),
                [](auto& task){ return task.status() == db::toloka::TaskStatus::Free; }),
            dbTasks.end());
        for (auto& dbTask : dbTasks) {
            auto offset = info.findTaskPos(dbTask.id());
            REQUIRE(offset < outputValues.size(), "Invalid task: " << dbTask.id());
            dbTask.setStatus(db::toloka::TaskStatus::Finished);
            dbTask.setOutputValues(outputValues[offset]);
            dbTask.setSolvedAt(NOW);
        }
        db::toloka::TaskGateway{*txn}.updatex(dbTasks);
        txn->commit();

        INFO() << "Saved " << name() << " suite (" << tolokaSuiteId << ")";
    }

private:
    struct SuiteWrapper {
        db::toloka::TolokaTaskSuite dbSuite;
        db::TIds dbTaskIds;

        size_t findTaskPos(db::TId dbTaskId) const
        {
            auto it = std::find(dbTaskIds.begin(), dbTaskIds.end(), dbTaskId);
            REQUIRE(it != dbTaskIds.end(), "Unknown task: " << dbTaskId);
            return std::distance(dbTaskIds.begin(), it);
        }
    };

    /// strong order - initialization of any field depends on the previous
    pgpool3::Pool& pool_;
    db::toloka::Platform platform_;
    db::toloka::TaskType type_;
    std::unordered_map<std::string, SuiteWrapper> tolokaIdToSuiteMap_;

    db::toloka::TolokaTasks loadInProgressTolokaTasks()
    {
        auto txn = pool_.slaveTransaction();
        using Tb = db::toloka::table::Task;
        auto dbTaskIds = db::toloka::TaskGateway{*txn}.loadIds(
            Tb::platform == platform_ &&
            Tb::type.equals(static_cast<int>(type_)) &&
            Tb::status.equals(db::toloka::TaskStatus::InProgress) &&
            Tb::knownSolutions.isNull());

        INFO() << "Loaded " << dbTaskIds.size() << " " << name()
               << " tasks IDs";
        auto dbTolokaTasks = db::toloka::TolokaTaskGateway{*txn}.loadByTaskIds(dbTaskIds);
        INFO() << "Loaded " << dbTolokaTasks.size() << " " << name()
               << " toloka tasks";
        return dbTolokaTasks;
    }

    db::toloka::TolokaTaskSuites
    loadSuitesByTasks(const db::toloka::TolokaTasks& dbTolokaTasks)
    {
        auto txn = pool_.slaveTransaction();
        std::unordered_set<db::TId> dbSuiteIds;
        for (auto& dbTolokaTask : dbTolokaTasks)
            dbSuiteIds.insert(dbTolokaTask.taskSuiteId());
        auto dbSuites = db::toloka::TolokaTaskSuiteGateway{*txn}.loadByIds(
            db::TIds{dbSuiteIds.begin(), dbSuiteIds.end()});
        INFO() << "Loaded " << dbSuites.size() << " " << name() << " suites";
        return dbSuites;
    }

    std::unordered_map<db::TId, SuiteWrapper> loadDbIdToSuiteMap()
    {
        std::unordered_map<db::TId, SuiteWrapper> dbIdToSuiteInfoMap;

        auto dbTolokaTasks = loadInProgressTolokaTasks();
        auto dbSuites = loadSuitesByTasks(dbTolokaTasks);

        for (auto& dbSuite : dbSuites) {
            auto dbId = dbSuite.id();
            dbIdToSuiteInfoMap.insert({dbId, {std::move(dbSuite), {}}});
        }

        for (auto& dbTolokaTask : dbTolokaTasks) {
            REQUIRE(dbIdToSuiteInfoMap.count(dbTolokaTask.taskSuiteId()),
                    "Unknown suite: " << dbTolokaTask.taskSuiteId());
            auto& suite = dbIdToSuiteInfoMap.at(dbTolokaTask.taskSuiteId());

            // Since we have not loaded golden tasks, some cells in suite.dbTaskIds
            // might remain empty. This is ok, because we do not need to handle
            // solved golden tasks each time.
            // TODO: improve processing of golden tasks.
            if (suite.dbTaskIds.size()
                < static_cast<size_t>(dbTolokaTask.taskIndex())) {
                suite.dbTaskIds.resize(dbTolokaTask.taskIndex());
            }
            suite.dbTaskIds[dbTolokaTask.taskIndex() - 1]
                = dbTolokaTask.taskId();
        }
        return dbIdToSuiteInfoMap;
    }

    std::unordered_map<std::string, SuiteWrapper> loadTolokaIdToSuiteMap()
    {
        auto dbIdToSuiteInfoMap = loadDbIdToSuiteMap();

        // translate suite key from Postgres (db::TId) to Toloka (std::string)
        std::unordered_map<std::string, SuiteWrapper> tolokaIdToSuiteInfoMap;
        for (auto& dbIdToSuiteInfo : dbIdToSuiteInfoMap) {
            auto tolokaId = dbIdToSuiteInfo.second.dbSuite.tolokaId();
            tolokaIdToSuiteInfoMap.insert(
                {tolokaId, std::move(dbIdToSuiteInfo.second)});
        }
        return tolokaIdToSuiteInfoMap;
    }
};

} // anonymous namespace

void checkCompletedTasks(db::toloka::Platform platform, pgpool3::Pool& pool, io::TolokaClient& client)
{
    TrafficLightDetectionHandler trafficLightDetectionHandler;
    ImageQualityHandler qualityHandler;
    ApprovementHandler approvementHandler;
    TaxonomyHandler taxonomyHandler;
    HouseNumberDetectionHandler houseNumberDetectionHandler;
    HouseNumberRecognitionHandler houseNumberRecognitionHandler;
    DetectionPairMatchHandler detectionPairMatchHandler;
    DetectionMissingOnFrameHandler detectionMissingOnFrameHandler;

    const std::vector<TolokaCompletedTasksHandler*>
        TOLOKA_COMPLETED_TASKS_HANDLERS = {&qualityHandler,
                                           &approvementHandler,
                                           &taxonomyHandler,
                                           &trafficLightDetectionHandler,
                                           &houseNumberDetectionHandler,
                                           &houseNumberRecognitionHandler,
                                           &detectionPairMatchHandler,
                                           &detectionMissingOnFrameHandler};

    for (auto& handler : TOLOKA_COMPLETED_TASKS_HANDLERS) {
        DbContext context{platform, pool, handler->type()};
        for (auto poolId : context.inProgressPoolIds()) {
            handler->handlePool(
                client, std::to_string(poolId),
                [&context](const std::string& tolokaSuiteId) {
                    return context.isSuiteInProgress(tolokaSuiteId);
                },
                [&context](const std::string& tolokaSuiteId,
                           const std::vector<std::string>& outputValues) {
                    context.saveSuite(tolokaSuiteId, outputValues);
                });
            INFO() << "Handled " << context.name() << " pool (" << poolId
                   << ")";
        }
        INFO() << "Done " << context.name() << " tasks";
    }
}

} // namespace toloka
} // namespace mrc
} // namespace maps
