#include "pqxx/transaction_base.hxx"
#include "toloka/platform.h"
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/create_new_tasks.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/make_batches.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_type_info_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/toloka_task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/toloka_task_suite_gateway.h>
#include <util/random/entropy.h>
#include <util/random/mersenne.h>

#include <boost/lexical_cast.hpp>

#include <algorithm>
#include <random>
#include <string>

namespace maps {
namespace mrc {
namespace toloka {
namespace {

static const std::vector<db::toloka::TaskType> TASK_TYPES{
    db::toloka::TaskType::ImageQualityClassification,
    db::toloka::TaskType::Approvement,
    db::toloka::TaskType::Taxonomy,
    db::toloka::TaskType::TrafficLightDetection,
    db::toloka::TaskType::HouseNumberDetection,
    db::toloka::TaskType::HouseNumberRecognition,
    db::toloka::TaskType::DetectionPairMatch,
    db::toloka::TaskType::DetectionMissingOnFrame,
};

class TaskTypeContext {
public:
    TaskTypeContext(db::toloka::Platform platform,
                    const common::TolokaTaskConfig& cfg,
                    pgpool3::Pool& pool,
                    toloka::io::TolokaClient& client,
                    db::toloka::TaskType type)
        : platform_(platform)
        , pool_(pool)
        , client_(client)
        , info_{loadOrCreateTaskTypeInfo(*pool_.slaveTransaction(), type)}
        , cfg_{cfg}
        , tolokaPool_{getTolokaPool()}
        , goldenTaskIds_{}
        , goldenTaskIdsShift_(0)
    {
    }

    db::toloka::TaskType type() { return info_.type(); }

    size_t suiteSize() const { return cfg_.suiteSize(); }

    size_t goldenTasksCount() const { return cfg_.goldenTasksCount(); }

    size_t completeSuiteSize() const
    {
        return suiteSize() + goldenTasksCount();
    }

    db::toloka::TaskTypeInfo
    loadOrCreateTaskTypeInfo(pqxx::transaction_base& txn, db::toloka::TaskType type)
    {
        if (auto taskTypeInfo =
                db::toloka::TaskTypeInfoGateway{txn}
                    .tryLoadById(platform_, type))
        {
            return taskTypeInfo.value();
        }
        return {platform_, db::toloka::TaskType{type}};
    }

    db::TIds loadNewTaskIds()
    {
        auto ids = db::toloka::TaskGateway{*pool_.slaveTransaction()}
                       .loadIdsByTypeStatus(platform_,
                                            info_.type(),
                                            db::toloka::TaskStatus::New);
        // implicit chronological order
        std::sort(ids.begin(), ids.end());
        INFO() << "Loaded " << ids.size() << " " << type() << " tasks";
        return ids;
    }

    db::TIds getGoldenTaskIds()
    {
        if (goldenTaskIds_.empty()) {
            goldenTaskIds_
                = db::toloka::TaskGateway{*pool_.slaveTransaction()}.loadIds(
                    db::toloka::table::Task::type.equals(
                        static_cast<int>(info_.type()))
                    && db::toloka::table::Task::knownSolutions.isNotNull());
            REQUIRE(goldenTasksCount() <= goldenTaskIds_.size(),
                    "Not enough golden tasks in database.");
        }
        db::TIds goldenTaskIds;
        for (size_t i = 0; i < goldenTasksCount(); ++i) {
            goldenTaskIds.push_back(goldenTaskIds_[goldenTaskIdsShift_++]);
            goldenTaskIdsShift_ %= goldenTaskIds_.size();
        }
        return goldenTaskIds;
    }

    void handleSuite(const db::TIds& taskIds)
    {
        if (info_.activePoolSize() >= cfg_.poolSize() ||
            tolokaPool_.status() == toloka::io::PoolStatus::Archived)
        {
            tolokaPool_ = createTolokaPool();
        }

        auto dbTasks = loadTasksByIds(taskIds);
        if (dbTasks.size() < completeSuiteSize() && !expired(dbTasks)) {
            INFO() << "Skipped " << type() << " tasks (" << taskIds.front()
                   << ":" << taskIds.back() << ")";
            return;
        }

        const auto tolokaSuite = postSuite(dbTasks);

        REQUIRE(dbTasks.size() == tolokaSuite.tasks().size(),
            "Posted and returned tasks count must be equal, but "
                << dbTasks.size() << " <= "  << tolokaSuite.tasks().size());

        db::toloka::TolokaTaskSuite dbSuite(platform_);
        dbSuite
            .setTolokaId(tolokaSuite.id())
            .setTolokaPoolId(boost::lexical_cast<db::TId>(tolokaSuite.poolId()))
            .setOverlap(tolokaSuite.overlap())
            .setCreatedAt(tolokaSuite.createdAt());

        info_.setActivePoolSize(info_.activePoolSize() + 1);

        auto txn = pool_.masterWriteableTransaction();
        db::toloka::TaskTypeInfoGateway{*txn}.upsert(info_);
        db::toloka::TolokaTaskSuiteGateway{*txn}.insert(dbSuite);
        db::toloka::TolokaTasks dbTolokaTasks;
        int taskIndex = 0;
        for (size_t i = 0; i < dbTasks.size(); ++i) {
            auto& dbTask = dbTasks[i];
            const auto& tolokaTask = tolokaSuite.tasks()[i];
            dbTolokaTasks.emplace_back(platform_, dbSuite.id(), ++taskIndex,
                                        dbTask.id(), tolokaTask.id());
            dbTask.setStatus(db::toloka::TaskStatus::InProgress);
            dbTask.setPostedAt(dbSuite.createdAt());
        }
        db::toloka::TolokaTaskGateway{*txn}.insert(dbTolokaTasks);
        db::toloka::TaskGateway{*txn}.updatex(dbTasks);
        txn->commit();

        INFO() << "Commited " << type() << " tasks (" << taskIds.front()
            << ":" << taskIds.back() << ")";
    }

private:
    /// strong order - initialization of any field depends on the previous
    db::toloka::Platform platform_;
    pgpool3::Pool& pool_;
    toloka::io::TolokaClient& client_;
    db::toloka::TaskTypeInfo info_;
    common::TolokaTaskConfig cfg_;
    toloka::io::Pool tolokaPool_;
    /// Caching golden tasks
    db::TIds goldenTaskIds_;
    size_t goldenTaskIdsShift_;

    toloka::io::Pool createTolokaPool()
    {
        toloka::io::PoolCreationParams params{
            client_.getPool(cfg_.templatePoolId())};
        params.setPrivateName("Toloka manager "
                              + chrono::formatIsoDateTime(
                                    chrono::TimePoint::clock::now()));
        auto tolokaPool = client_.createPool(params);
        INFO() << "Created " << type() << " pool (" << tolokaPool.id() << ")";

        info_.setActivePoolId(boost::lexical_cast<db::TId>(tolokaPool.id()))
            .setActivePoolSize(0);

        auto txn = pool_.masterWriteableTransaction();
        db::toloka::TaskTypeInfoGateway{*txn}.upsert(info_);
        txn->commit();
        return tolokaPool;
    }

    toloka::io::Pool getTolokaPool()
    {
        return (info_.activePoolId()
                && info_.activePoolSize() < cfg_.poolSize())
                   ? client_.getPool(std::to_string(info_.activePoolId()))
                   : createTolokaPool();
    }

    db::toloka::Tasks loadTasksByIds(const db::TIds& ids)
    {
        auto tasks
            = db::toloka::TaskGateway{*pool_.slaveTransaction()}.loadByIds(
                ids);
        std::shuffle(tasks.begin(), tasks.end(), TMersenne<ui64>(Seed()));
        return tasks;
    }

    toloka::io::TaskSuite postSuite(const db::toloka::Tasks& dbTasks)
    {
        toloka::io::TaskSuiteItems items;
        for (const auto& dbTask : dbTasks) {
            items.emplace_back(
                io::TaskSuiteItem::fromInputValues(
                    json::Value::fromString(dbTask.inputValues())
                )
            );
            auto& lastItem = items.back();
            if (dbTask.knownSolutions()) {
                auto valueKnownSolutions
                    = json::Value::fromString(*dbTask.knownSolutions());
                REQUIRE(valueKnownSolutions.isArray(),
                        "Invalid known_solutions value should be json array");
                toloka::io::KnownSolutions knownSolutions;
                for (const auto& valueKnownSolution : valueKnownSolutions) {
                    knownSolutions.emplace_back(
                        valueKnownSolution["output_values"],
                        valueKnownSolution["correctness_weight"]
                            .as<double>());
                }
                lastItem.setKnownSolutions(std::move(knownSolutions));
            }
            if (dbTask.messageOnUnknownSolution()) {
                lastItem.setMessageOnUnknownSolution(
                    *dbTask.messageOnUnknownSolution());
            }
        }

        auto tolokaSuite = client_.postTaskSuite(
            {tolokaPool_.id(), items},
            {{"allow_defaults", "true"}, {"open_pool", "true"}});
        INFO() << "Posted " << type() << " suite (" << tolokaSuite.id()
               << ")";
        return tolokaSuite;
    }

    bool expired(const db::toloka::Tasks& dbTasks)
    {
        static const std::chrono::hours TIMEOUT{24};
        REQUIRE(!dbTasks.empty(), "Empty batch");
        // Searching for oldest task, skipping golden tasks
        auto oldest = dbTasks.begin();
        for (auto task = dbTasks.begin(); task != dbTasks.end(); ++task) {
            bool isRegularTask = !task->knownSolutions();
            bool olderTask = task->createdAt() < oldest->createdAt();
            if (isRegularTask && (oldest->knownSolutions() || olderTask))
                oldest = task;
        }
        auto age = chrono::TimePoint::clock::now() - oldest->createdAt();
        return age > TIMEOUT;
    }
};

} // anonymous namespace

void createNewTasks(db::toloka::Platform platform,
                    const common::Config& cfg,
                    pgpool3::Pool& pool,
                    toloka::io::TolokaClient& client)
{
    const auto& platformConfig = cfg.crowdPlatformConfig(platform);
    for (auto taskType : TASK_TYPES) {
        if (auto taskConfig = platformConfig.taskConfig(taskType)) {

            TaskTypeContext context{platform, taskConfig.value(), pool, client, taskType};
            auto ids = context.loadNewTaskIds();
            if (!ids.empty()) {
                for (auto& batch :
                    maps::common::makeBatches(ids, context.suiteSize())) {
                    db::TIds batchIds({batch.begin(), batch.end()});
                    auto goldenIds = context.getGoldenTaskIds();
                    if (!goldenIds.empty()) {
                        INFO() << "Adding " << goldenIds.size() << " golden "
                            << context.type() << " tasks to batch";
                        batchIds.insert(batchIds.end(), goldenIds.begin(),
                                        goldenIds.end());
                    }
                    context.handleSuite(batchIds);
                }
            }
            INFO() << "Done " << taskType << " tasks";

        }
    }
}

} // namespace toloka
} // namespace mrc
} // namespace maps
