#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/detection.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/contains.h>
#include <maps/libs/log8/include/log8.h>

#include <boost/range/algorithm_ext/erase.hpp>

#include <cmath>
#include <fstream>

namespace maps {
namespace mrc {
namespace toloka {

namespace {

// UserID and box
using UserBox = std::pair<std::string, geolib3::BoundingBox>;
using BoxesGroup = std::vector<UserBox>;

double diff(double a, double b)
{
    return (a > b) ? (a - b) : (b - a);
}

/**
 * Match given box with a group of boxes.
 * Returns true if \p box overlaps with every box in the \p group
 * with relative difference by each coordinate not more than 30%.
 */
bool boxMatchesGroup(const geolib3::BoundingBox& box,
                           const BoxesGroup& group)
{
    // прямоугольник считается принадлежащим группе, если
    // каждая из его четырех координат x_min, x_max, y_min, y_max
    // отличается от коордитах всех прямоугольников в группе не более,
    // на 0.5% от размеров изображения. Для стандартного изображения
    // в разрешении FullHD это примерно 5-10 пикселей, и при этом
    // ширина и высота прямоугольника отличается не более, чем на 30%
    // от соответствующих значений всех прямоугольников в группе.
    constexpr double THRESH_DIFF_SIZE = 0.005;
    constexpr double MAX_RELATIVE_COORD_DIFF = 0.3;

    for (const auto& userAndBox : group) {
        const auto& other = userAndBox.second;
        auto intersection = geolib3::intersection(box, other);
        if (!intersection) {
            return false;
        }

        double width = std::max(box.maxX() - box.minX(),
                                other.maxX() - other.minX());
        double height = std::max(box.maxY() - box.minY(),
                                 other.maxY() - other.minY());

        double dxMin = diff(box.minX(), other.minX());
        double dxMax = diff(box.maxX(), other.maxX());
        double dyMin = diff(box.minY(), other.minY());
        double dyMax = diff(box.maxY(), other.maxY());

        if ((dxMin > THRESH_DIFF_SIZE
             && dxMin / width > MAX_RELATIVE_COORD_DIFF)
            || (dxMax > THRESH_DIFF_SIZE
                && dxMax / width > MAX_RELATIVE_COORD_DIFF)
            || (dyMin > THRESH_DIFF_SIZE
                && dyMin / height > MAX_RELATIVE_COORD_DIFF)
            || (dyMax > THRESH_DIFF_SIZE
                && dyMax / height > MAX_RELATIVE_COORD_DIFF)) {
            return false;
        }
    }
    return true;
}

/**
 * Tells if the given group contains a box, selected by the given user.
 */
bool groupHasBoxFromUser(const BoxesGroup& group,
                               const std::string& uid)
{
    return std::find_if(group.begin(), group.end(),
                        [&](const UserBox& pair) {
                            return pair.first == uid;
                        }) != group.end();
}

void retainGroupsWithMajority(std::vector<BoxesGroup>& groups,
                              size_t majority)
{
    boost::range::remove_erase_if(groups, [&](const BoxesGroup& group) {
        return group.size() < majority;
    });
}

/**
 * Collapse group of boxes into a single box.
 * Each coordinate (left, right, top, bottom) of the resulting box
 * is calculated as the median of the corresponding coordinates
 * of all boxes in the group.
 */
geolib3::BoundingBox mergeBoxesGroup(const BoxesGroup& group)
{
    REQUIRE(!group.empty(), "Empty boxes group");

    auto median = [](std::vector<double>& vec) -> double {
        std::nth_element(vec.begin(), vec.begin() + vec.size() / 2,
                         vec.end());
        return vec[vec.size() / 2];
    };

    size_t size = group.size();
    std::vector<double> minX(size), maxX(size), minY(size), maxY(size);
    for (size_t i = 0; i < size; ++i) {
        minX[i] = group[i].second.minX();
        maxX[i] = group[i].second.maxX();
        minY[i] = group[i].second.minY();
        maxY[i] = group[i].second.maxY();
    }

    return geolib3::BoundingBox(
        geolib3::Point2(median(minX), median(minY)),
        geolib3::Point2(median(maxX), median(maxY))
    );
}

/**
 * Merge groups of boxes by collapsing each group into one box.
 * If a group contains less than \p minBoxesInGroup boxes,
 * the group is 'unconfirmed' and no box is generated for this group.
 */
std::vector<geolib3::BoundingBox> mergeBoxesGroups(const std::vector<BoxesGroup>& groups)
{
    std::vector<geolib3::BoundingBox> result;
    result.reserve(groups.size());
    for (const auto& group : groups) {
        result.push_back(mergeBoxesGroup(group));
    }
    return result;
}

/**
 * Merge Toloka results for a single task received from multiple users
 */
Detection::TaskOutput mergeSingleTaskResults(
    const Detection::AssignmentResults& results,
    size_t taskIdx)
{
    std::vector<BoxesGroup> boxGroups;

    // Group boxes marked by different users according to intersections
    for (const auto& result : results) {
        REQUIRE(result.outputs.size() > taskIdx, "No result for task #"
                                                     << taskIdx);
        const auto& uid = result.userId;
        const auto& bboxes = result.outputs[taskIdx].bboxes();

        std::vector<geolib3::BoundingBox> unmatched;

        // Try to intersect every new box with every group
        for (const auto& bbox : bboxes) {
            bool matched = false;
            for (auto& group : boxGroups) {
                if (boxMatchesGroup(bbox, group)
                    && !groupHasBoxFromUser(group, uid)) {
                    group.push_back({uid, bbox});
                    matched = true;
                    break;
                }
            }
            // Create new group if box did not match any existing group
            if (!matched) {
                boxGroups.push_back({{uid, bbox}});
            }
        }
    }
    retainGroupsWithMajority(boxGroups, results.size() / 2 + 1);

    auto bboxes = mergeBoxesGroups(boxGroups);

    DetectionResult result = bboxes.empty() ?
                                   DetectionResult::IsEmpty
                                 : DetectionResult::IsNotEmpty;

    return Detection::TaskOutput{result, std::move(bboxes)};
}

} // anonymous namespace

Detection::TaskSuiteResult Detection::mergeTaskSuiteResults(
    const Detection::AssignmentResults& assignmentResults)
{
    Detection::TaskSuiteResult tsResult;

    REQUIRE(!assignmentResults.empty(), "Empty task suite results");
    size_t tasksCount = assignmentResults[0].inputs.size();

    for (size_t i = 0; i < tasksCount; ++i) {
        Detection::TaskInput input = assignmentResults[0].inputs[i];
        Detection::TaskOutput output = mergeSingleTaskResults(assignmentResults, i);

        tsResult.taskResults.push_back({input, output});
    }

    return tsResult;
}

void Detection::evaluateAssignments(
    const io::TolokaClient& /*tolokaClient*/,
    const Detection::TaskSuiteResult& /*tsResult*/)
{
    // в этой функции ничего не происходит, так как все задания
    // автоматически считаются принятыми, а контроль качества выполнения
    // настроен на стороне Толоки на основе контрольных заданий.
    return;
}

} // toloka
} // mrc
} // maps
