#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/conversion.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/mrc/toloka_client/client.h>

#include <algorithm>
#include <ostream>

namespace maps {

namespace geolib3 {

inline std::ostream& operator<<(std::ostream& out, const BoundingBox& box)
{
    return out << "[[" << box.minX() << "," << box.minY() << "]"
               << ",[" << box.maxX() << "," << box.maxY() << "]]";
}

} // namespace geolib3

namespace mrc {
namespace toloka {

template <typename Container>
void sortUnique(Container& cont)
{
    std::sort(cont.begin(), cont.end());
    const auto end = std::unique(cont.begin(), cont.end());
    cont.resize(std::distance(cont.begin(), end));
}

// Return pair of iterators [first, last) representing the longest equal range
template <typename Iterator>
std::pair<Iterator, Iterator> longestEqualRange(Iterator first, Iterator last)
{
    size_t maxSize = 0;
    auto maxFirst = last;
    auto maxLast = last;

    while (first != last) {
        auto pair = std::equal_range(first, last, *first);
        size_t size = std::distance(pair.first, pair.second);
        if (size > maxSize) {
            maxSize = size;
            maxFirst = pair.first;
            maxLast = pair.second;
        }
        first = pair.second;
    }
    return {maxFirst, maxLast};
}

template <typename IdToTaskSuite, typename GetInputValues>
IdToTaskSuite load(const io::TolokaClient& tolokaClient,
                   const std::string& poolId,
                   GetInputValues getInputValues)
{
    INFO() << "Loading task suites";
    IdToTaskSuite resultMap;
    io::Filter filter;
    filter.byPoolId(poolId);
    auto resp = tolokaClient.getTaskSuites(std::move(filter));

    for (const auto& ts : resp.taskSuites()) {
        typename IdToTaskSuite::mapped_type taskSuite;
        taskSuite.id = ts.id();
        taskSuite.poolId = ts.poolId();
        taskSuite.overlap = ts.overlap();
        taskSuite.tasks = getInputValues(ts.tasks());
        resultMap.emplace(ts.id(), std::move(taskSuite));
    }
    return resultMap;
}

template <typename TaskSuiteIdToResults,
          typename GetInputValues,
          typename GetOutputValues>
TaskSuiteIdToResults load(const io::TolokaClient& tolokaClient,
                          const std::string& poolId,
                          GetInputValues getInputValues,
                          GetOutputValues getOutputValues)
{
    INFO() << "Loading task suite results";
    TaskSuiteIdToResults resultMap;
    io::Filter filter;
    filter.byPoolId(poolId);
    auto resp = tolokaClient.getAssignments(std::move(filter));

    for (const auto& assignment : resp.assignments()) {
        // Keep only the assignments which have output values,
        // i.e. skip Active/Skipped/Expired assignments
        if (assignment.status() != io::AssignmentStatus::Accepted
            && assignment.status() != io::AssignmentStatus::Submitted
            && assignment.status() != io::AssignmentStatus::Rejected) {
            continue;
        }

        typename TaskSuiteIdToResults::mapped_type::value_type result;
        result.taskSuiteId = assignment.taskSuiteId();
        result.assignmentId = assignment.id();
        result.status = assignment.status();
        result.userId = assignment.userId();
        result.inputs = getInputValues(assignment.tasks());
        result.outputs = getOutputValues(assignment.solutions());

        resultMap[assignment.taskSuiteId()].push_back(std::move(result));
    }
    return resultMap;
}

template <typename TaskInput>
std::vector<TaskInput> getInputValues(const io::TaskSuiteItems& items)
{
    std::vector<TaskInput> taskInputs;
    for (const auto& item : items) {
        taskInputs.push_back(parseJson<TaskInput>(item.inputValues()));
    }
    return taskInputs;
}

template <typename TaskOutput>
std::vector<TaskOutput> getOutputValues(const io::AssignmentSolutions& items)
{
    std::vector<TaskOutput> taskOutputs;
    for (const auto& item : items) {
        taskOutputs.push_back(parseJson<TaskOutput>(item.outputValues()));
    }
    return taskOutputs;
}

template <typename TaskType>
typename TaskType::IdToTaskSuite loadTaskSuites(
    const io::TolokaClient& tolokaClient,
    const std::string& poolId)
{
    return load<typename TaskType::IdToTaskSuite>(
        tolokaClient, poolId, getInputValues<typename TaskType::TaskInput>);
}

template <typename TaskType>
typename TaskType::TaskSuiteIdToResults loadTaskSuitesResults(
    const io::TolokaClient& tolokaClient,
    const std::string& poolId)
{
    return load<typename TaskType::TaskSuiteIdToResults>(
        tolokaClient, poolId,
        getInputValues<typename TaskType::TaskInput>,
        getOutputValues<typename TaskType::TaskOutput>);
}

/**
 * Merge single task suite results received from multiple users
 */
template <typename TaskType>
typename TaskType::TaskSuiteResult mergeTaskSuiteResultsImpl(
    const typename TaskType::AssignmentResults& assignmentResults)
{
    typename TaskType::TaskSuiteResult tsResult;
    REQUIRE(!assignmentResults.empty(), "Empty task suite results");
    const size_t tasksCount = assignmentResults[0].inputs.size();
    for (size_t i = 0; i < tasksCount; ++i) {
        auto input = assignmentResults[0].inputs[i];
        auto output = TaskType::mergeSingleTaskResults(assignmentResults, i,
                                                       tsResult.userIdToStat);
        tsResult.taskResults.push_back(
            typename TaskType::TaskResult{std::move(input), std::move(output)});
    }
    return tsResult;
}

template <typename TaskType>
typename TaskType::TaskSuiteIdToResultMap mergeTasksResults(
    const typename TaskType::IdToTaskSuite& idToTaskSuite,
    const typename TaskType::TaskSuiteIdToResults& idToAssignmentResults)
{
    typename TaskType::TaskSuiteIdToResultMap tsResults;
    for (auto& idAndResults : idToAssignmentResults) {
        const auto& id = idAndResults.first;
        const auto& assignmentResults = idAndResults.second;
        auto taskSuiteItr = idToTaskSuite.find(id);
        if (taskSuiteItr == idToTaskSuite.end()) {
            // Skip results for unknown task suite
            WARN() << "Skip unknown task suite: " << id;
            continue;
        }
        // This can only happen if running on an unfinished pool
        if (assignmentResults.size() < taskSuiteItr->second.overlap) {
            INFO() << "Results not ready for task suite " << id;
            continue;
        }
        INFO() << "Merge results for task suite " << id;
        tsResults.insert({id, TaskType::mergeTaskSuiteResults(assignmentResults)});
    }
    return tsResults;
}

template <typename TaskType>
void evaluateAssignmentsImpl(
    const io::TolokaClient& tolokaClient,
    const typename TaskType::TaskSuiteResult& tsResult)
{
    constexpr double THRESHOLD_PRECISION = 0.7;
    for (const auto& userAndStat : tsResult.userIdToStat) {
        const auto& stat = userAndStat.second;
        if (stat.assignmentStatus != io::AssignmentStatus::Submitted) {
            continue;
        }
        auto precision = (double)stat.correctCount / stat.tasksCount;
        if (precision < THRESHOLD_PRECISION) {
            INFO() << "REJECT assignment " << stat.assignmentId
                   << " from user " << stat.userId << ", precision "
                   << precision;
            tolokaClient.rejectAssignment(
                stat.assignmentId,
                "Некоторые задания выполнены неверно");
        }
        else {
            INFO() << "ACCEPT assignment " << stat.assignmentId
                   << " from user " << stat.userId;
            tolokaClient.acceptAssignment(stat.assignmentId);
        }
    }
}

} // namespace toloka
} // namespace mrc
} // namespace maps
