#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/taxonomy.h>
#include <maps/libs/geolib/include/point.h>

namespace maps {
namespace mrc {
namespace toloka {
namespace test {
namespace {

const std::string TS_1 = "task-suite-id-1";
const std::string TS_2 = "task-suite-id-2";

const std::string POOL_ID = "1";

const std::string USER_1 = "user-id-1";
const std::string USER_2 = "user-id-2";
const std::string USER_3 = "user-id-3";

const std::string SOURCE_1 = "source-1";
const std::string SOURCE_2 = "source-2";
const std::string SOURCE_3 = "source-3";
const std::string SOURCE_4 = "source-4";

constexpr size_t OVERLAP = 3;

const common::ImageBox bbox(100, 100, 200, 200);

const std::string MARKING_1_18_1 = "1.18.1_Russian_road_marking";
const std::string MARKING_1_18_2 = "1.18.2_Russian_road_marking";
const std::string MARKING_1_18_3 = "1.18.3_Russian_road_marking";
const std::string MARKING_1_18_4 = "1.18.4_Russian_road_marking";
const std::string UNKNOWN = "unknown";

} // anonymous namespace

TEST(taxonomy_tests, merge_task_suites)
{
    Taxonomy::IdToTaskSuite idToTaskSuite{
        {TS_1,
         {TS_1,
          POOL_ID,
          OVERLAP,
          {Taxonomy::TaskInput{SOURCE_1, bbox}, Taxonomy::TaskInput{SOURCE_2, bbox},
           Taxonomy::TaskInput{SOURCE_3, bbox}}}},
        {TS_2, {TS_2, POOL_ID, OVERLAP, {Taxonomy::TaskInput{SOURCE_4, bbox}}}}};

    Taxonomy::TaskSuiteIdToResults taskSuiteIdToResults{
        /** In task suite 1:
         *  task1 - all users agree
         *  task2 - 2 users vs 1 user
         *  task3 - 3 users give 3 different answers
         */
        {TS_1,
         {{// Result from user 1
           TS_1,
           "unused",
           io::AssignmentStatus::Submitted,
           USER_1,
           {Taxonomy::TaskInput{SOURCE_1, bbox}, Taxonomy::TaskInput{SOURCE_2, bbox},
            Taxonomy::TaskInput{SOURCE_3, bbox}},
           {
               Taxonomy::TaskOutput{MARKING_1_18_3}, Taxonomy::TaskOutput{MARKING_1_18_1},
               Taxonomy::TaskOutput{MARKING_1_18_4},
           }},
          {// Result from user 2
           TS_1,
           "unused",
           io::AssignmentStatus::Submitted,
           USER_2,
           {Taxonomy::TaskInput{SOURCE_1, bbox}, Taxonomy::TaskInput{SOURCE_2, bbox},
            Taxonomy::TaskInput{SOURCE_3, bbox}},
           {
               Taxonomy::TaskOutput{MARKING_1_18_3}, Taxonomy::TaskOutput{MARKING_1_18_1},
               Taxonomy::TaskOutput{UNKNOWN},
           }},
          {// Result from user 3
           TS_1,
           "unused",
           io::AssignmentStatus::Submitted,
           USER_3,
           {Taxonomy::TaskInput{SOURCE_1, bbox}, Taxonomy::TaskInput{SOURCE_2, bbox},
            Taxonomy::TaskInput{SOURCE_3, bbox}},
           {
               Taxonomy::TaskOutput{MARKING_1_18_3}, Taxonomy::TaskOutput{MARKING_1_18_2},
               Taxonomy::TaskOutput{MARKING_1_18_1},
           }}}},

        /** In task suite 2:
         *  Only 2 users accomplished the assignment, result is not ready
         */
        {TS_2,
         {{TS_2,
           "unused",
           io::AssignmentStatus::Submitted,
           USER_1,
           {Taxonomy::TaskInput{SOURCE_4, bbox}},
           {Taxonomy::TaskOutput{MARKING_1_18_1}}},
          {TS_2,
           "unused",
           io::AssignmentStatus::Submitted,
           USER_2,
           {Taxonomy::TaskInput{SOURCE_4, bbox}},
           {Taxonomy::TaskOutput{MARKING_1_18_1}}}}}};

    auto tsResults = mergeTasksResults<Taxonomy>(idToTaskSuite, taskSuiteIdToResults);

    // Validate taxonomy results
    // Results are available only for task suite 1, but not for
    // task suite 2, where not enough users have accomplished the assignments
    EXPECT_TRUE(tsResults.size() == 1) << "Wrong number of results";
    const auto& suiteResult = tsResults.begin()->second;
    const auto& taskResults = suiteResult.taskResults;

    EXPECT_TRUE(taskResults.size() == 3) << "Wrong number of tasks in task suite 1";

    EXPECT_EQ(taskResults[0].output.result(), MARKING_1_18_3);
    EXPECT_EQ(taskResults[1].output.result(), MARKING_1_18_1);
    EXPECT_EQ(taskResults[2].output.result(), "");

    // Validate user statistics
    const auto& stat1 = suiteResult.userIdToStat.at(USER_1);
    const auto& stat2 = suiteResult.userIdToStat.at(USER_2);
    const auto& stat3 = suiteResult.userIdToStat.at(USER_3);

    EXPECT_EQ(stat1.tasksCount, 3u);
    EXPECT_EQ(stat1.correctCount, 2);

    EXPECT_EQ(stat2.tasksCount, 3u);
    EXPECT_EQ(stat2.correctCount, 2);

    EXPECT_EQ(stat3.tasksCount, 3u);
    EXPECT_EQ(stat3.correctCount, 1);
}

} // test
} // toloka
} // mrc
} // maps
