#include "fixture.h"
#include "toloka/platform.h"

#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/cancel_tasks.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/check_completed_tasks.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/create_new_tasks.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/toloka_manager_cron_jobs/lib/include/delete_free_tasks.h>

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/toloka_task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/toloka_task_suite_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_type_info_gateway.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/http/include/test_utils.h>
#include <maps/libs/json/include/value.h>
#include <yandex/maps/mds/mds.h>
#include <yandex/maps/mrc/toloka_client/client.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

#include <library/cpp/testing/gmock_in_unittest/gmock.h>
#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>


namespace maps::mrc::toloka::tests {

namespace {

const std::string DATA_PATH = "data.sql";

class DbFixture : public testing::Test,
                  public unittest::WithUnittestConfig<unittest::DatabaseFixture>
{
public:
    DbFixture()
    {
        auto txn = pool().masterWriteableTransaction();
        db::toloka::TaskTypeInfoGateway(*txn).remove(sql_chemistry::AnyFilter{});
        txn->commit();
    }
};

class DbAndTolokaFixture : public DbFixture, public TolokaStubFixture {
public:
    DbAndTolokaFixture() = default;
};

class FilledFixture
    : public testing::Test,
      public unittest::WithUnittestConfig<unittest::MdsStubFixture,
                                          unittest::DatabaseFixture>,
      public TolokaStubFixture {
public:
    FilledFixture()
    {
        auto txn = pool().masterWriteableTransaction();
        db::toloka::TaskTypeInfoGateway(*txn).remove(sql_chemistry::AnyFilter{});
        txn->exec(maps::common::readFileToString(SRC_(DATA_PATH)));

        // Add task suite and task with a fresh solved_at date
        auto now = chrono::TimePoint(std::chrono::system_clock::now());
        auto solvedAt
            = chrono::formatIsoDateTime(now - std::chrono::hours(6));

        // clang-format off
        txn->exec(
            "INSERT INTO toloka_mgr.toloka_task_suite (id, toloka_id, toloka_pool_id, overlap, created_at, solved_at)\n"
            "VALUES (8, 'fca8ee8d-78f2-47bf-9738-a6620da4b021', 2000, 3, '2017-01-03 09:00:00', '" + solvedAt + "');\n"

            "INSERT INTO toloka_mgr.task (id, txn_id, type_id, status, input_values, output_values, overlap, created_at, posted_at, solved_at)\n"
            "VALUES (14, 0, 3, 'free', 'task-14-input', null, 3, '2017-01-03 09:00:00', '2017-01-03 12:00:00', '" + solvedAt + "');\n"

            "INSERT INTO toloka_mgr.toloka_task (task_suite_id, task_index, task_id)\n"
            "VALUES (8, 1, 14);\n"
        );
        // clang-format on
        txn->commit();
    }
};

} // anonymous namespace

TEST_F(FilledFixture, test_delete_free_tasks)
{
    {
        auto mdsClient = config().makePublicMdsClient();
        deleteFreeTasks(db::toloka::Platform::Toloka, pool(), mdsClient);
    }

    auto txn = pool().slaveTransaction();
    db::toloka::TaskGateway gtw{*txn};

    // Check remaining ImageQualityClassification tasks
    auto qualityTaskIds = gtw.loadIdsByType(db::toloka::Platform::Toloka, db::toloka::TaskType::ImageQualityClassification);
    EXPECT_EQ(qualityTaskIds.size(), 10u);

    /**
     * The remaining ImageQualityClassification tasks should be:
     * - task 20 is in status New
     * - task 21 is in status InProgress
     * - tasks 26, 27, 28 is in status New and contain known solutions
     */
    db::TIds expectedQualityTaskIds{1, 2, 5, 6, 14, 20, 21, 26, 27, 28};
    std::sort(qualityTaskIds.begin(), qualityTaskIds.end());

    EXPECT_EQ(qualityTaskIds, expectedQualityTaskIds);

    // Check remaining Approvement tasks
    auto approvementTaskIds = gtw.loadIdsByType(db::toloka::Platform::Toloka, db::toloka::TaskType::Approvement);
    EXPECT_EQ(approvementTaskIds.size(), 8u);

    /**
     * The only remaining Approvement tasks should be:
     * - task 23 is in status New
     * - task 24 is in status InProgress
     * - tasks 29, 30, 31 is in status New and contain known solutions
     */
    db::TIds expectedApprovementTaskIds{3, 4, 7, 23, 24, 29, 30, 31};
    std::sort(approvementTaskIds.begin(), approvementTaskIds.end());

    EXPECT_EQ(approvementTaskIds, expectedApprovementTaskIds);
}

TEST_F(FilledFixture, test_create_new_tasks)
{
    const auto NOW = chrono::TimePoint::clock::now();
    const auto TASK_ID = db::TId{1};

    size_t suits;
    {
        auto txn = pool().slaveTransaction();
        db::toloka::TaskGateway gtw{*txn};
        EXPECT_EQ(
            6u, gtw.loadIdsByStatus(db::toloka::Platform::Toloka, db::toloka::TaskStatus::New).size());
        suits = db::toloka::TolokaTaskSuiteGateway{*txn}.count();
        auto task = gtw.loadById(TASK_ID);
        EXPECT_TRUE(task.status() == db::toloka::TaskStatus::New);
        EXPECT_TRUE(!task.postedAt());
    }
    createNewTasks(db::toloka::Platform::Toloka, config(), pool(), tolokaClient());

    db::TId signDetectionActivePoolId = 0;
    {
        auto txn = pool().slaveTransaction();
        db::toloka::TaskGateway gtw{*txn};
        EXPECT_EQ(0u, gtw.loadIdsByStatus(db::toloka::Platform::Toloka, db::toloka::TaskStatus::New).size());
        EXPECT_EQ(suits + 2, db::toloka::TolokaTaskSuiteGateway{*txn}.count());
        auto task = gtw.loadById(TASK_ID);
        EXPECT_TRUE(task.status() == db::toloka::TaskStatus::InProgress);
        EXPECT_TRUE(*task.postedAt() > NOW);
        signDetectionActivePoolId
            = db::toloka::TaskTypeInfoGateway{*txn}.loadById(
                db::toloka::Platform::Toloka,
                db::toloka::TaskType::ImageQualityClassification)
                  .activePoolId();
    }

    {
        auto txn = pool().masterWriteableTransaction();
        // clang-format off
        txn->exec(
            "INSERT INTO toloka_mgr.task (txn_id, type_id, status, input_values, overlap, created_at) VALUES\n"
            "(0, 3, 'new', '{\"input\": 101}', 3, '2017-01-01 09:00:00'),\n"
            "(0, 3, 'new', '{\"input\": 102}', 3, '2017-01-01 09:00:00'),\n"
            "(0, 3, 'new', '{\"input\": 103}', 3, '2017-01-01 09:00:00');\n"
        );
        // clang-format on
        txn->commit();
    }
    createNewTasks(db::toloka::Platform::Toloka, config(), pool(), tolokaClient());

    {
        auto txn = pool().slaveTransaction();
        EXPECT_NE(
            signDetectionActivePoolId,
            db::toloka::TaskTypeInfoGateway{*txn}.loadById(
                db::toloka::Platform::Toloka,
                db::toloka::TaskType::ImageQualityClassification)
                .activePoolId());
    }
}

TEST_F(DbAndTolokaFixture, test_check_completed_tasks)
{
    {
        auto txn = pool().masterWriteableTransaction();
        // clang-format off
        txn->exec(R"(

            INSERT INTO toloka_mgr.toloka_task_suite (id, toloka_id, toloka_pool_id, overlap, created_at, solved_at)
            VALUES (2, 'task-suite-123', 1000, 3, '2017-01-01 09:00:00', null);

            INSERT INTO toloka_mgr.task_type_info (type_id)
                VALUES (3);

            INSERT INTO toloka_mgr.task (id, txn_id, type_id, status, input_values, output_values, overlap, created_at, posted_at, solved_at)
            VALUES (7, 0, 3, 'in-progress', '{"image": "http://image.com/image1", "bbox": [[100,100],[200,200]]}', null, 3, '2017-01-01 09:00:00', '2017-01-02 09:00:00', null);

            INSERT INTO toloka_mgr.toloka_task (task_suite_id, task_index, task_id)
            VALUES (2, 1, 7);

        )");
        // clang-format on
        txn->commit();
    }

    {
        auto txn = pool().masterWriteableTransaction();
        db::toloka::TaskGateway gtw{*txn};
        EXPECT_EQ(
            1u, gtw.loadIdsByStatus(db::toloka::Platform::Toloka, db::toloka::TaskStatus::InProgress)
                   .size());
    }

    checkCompletedTasks(db::toloka::Platform::Toloka, pool(), tolokaClient());

    {
        auto txn = pool().masterWriteableTransaction();
        db::toloka::TaskGateway gtw{*txn};
        EXPECT_EQ(
            0u, gtw.loadIdsByStatus(db::toloka::Platform::Toloka, db::toloka::TaskStatus::InProgress)
                   .size());
    }
}


class SuiteWithTwoTasksFixture : public DbFixture
{
public:

    SuiteWithTwoTasksFixture()
    {
        auto txn = pool().masterWriteableTransaction();

        db::toloka::TaskTypeInfoGateway{*txn}.insert(
            db::toloka::TaskTypeInfo{db::toloka::Platform::Toloka, db::toloka::TaskType::Approvement}
        );

        const size_t tasksCount = 2;

        for (size_t i = 0; i < tasksCount; ++i) {
            tasks.push_back(db::toloka::Task(db::toloka::Platform::Toloka)
                                .setStatus(db::toloka::TaskStatus::InProgress)
                                .setType(db::toloka::TaskType::Approvement));
        }

        db::toloka::TaskGateway{*txn}.insertx(tasks);

        suites.push_back(
            db::toloka::TolokaTaskSuite{db::toloka::Platform::Toloka}
                .setTolokaId("toloka_suite_id")
        );
        db::toloka::TolokaTaskSuiteGateway{*txn}.insert(suites);

        for (size_t i = 0; i < tasks.size(); ++i) {
            tolokaTasks.emplace_back(
                db::toloka::Platform::Toloka,
                suites[0].id(),
                i + 1,
                tasks[i].id(),
                "toloka_id_" + std::to_string(i));
        }
        db::toloka::TolokaTaskGateway{*txn}.insert(tolokaTasks);

        txn->commit();
    }

    io::TolokaClient tolokaClient()
    {
        const auto& crowdPlatformConfig =
            config().crowdPlatformConfig(db::toloka::Platform::Toloka);
        return toloka::io::TolokaClient{
            crowdPlatformConfig.host(), crowdPlatformConfig.authHeader()};
    }

    http::URL tolokaUrl()
    {
        return http::URL{"https://" +
            config().crowdPlatformConfig(db::toloka::Platform::Toloka).host()};
    }

    db::toloka::Tasks tasks;
    db::toloka::TolokaTaskSuites suites;
    db::toloka::TolokaTasks tolokaTasks;


    static constexpr std::string_view MOCK_TASK_RESPONSE = R"({
            "id": "2",
            "pool_id": "1",
            "input_values": {
                "source": "http://image.com/image2"
            },
            "overlap": 3,
            "infinite_overlap": false,
            "created": "2016-09-07T09:20:10"
        })";

    static constexpr std::string_view MOCK_TASKS_SUITE_RESPONSE = R"(
        {
            "id": "task-suite-123",
            "pool_id": "1000",
            "tasks": [
                {
                    "input_values": {"image": "http://image.com/image1", "bbox": [[100,100],[200,200]]},
                    "known_solutions": [
                        {
                            "correctness_weight": 0.95,
                            "output_values": {
                                "problem": "not-signs"
                            }
                        }
                    ],
                    "message_on_unknown_solution": "Слон черный"
                }
            ],
            "overlap": 3,
            "created": "2016-09-07T09:20:10"
        }
    )";
};


TEST_F(SuiteWithTwoTasksFixture, test_cancel_one_task)
{
    {
        auto txn = pool().masterWriteableTransaction();
        tasks[0].setStatus(db::toloka::TaskStatus::Cancelling);
        db::toloka::TaskGateway{*txn}.updatex(tasks);
        txn->commit();
    }

    auto client = tolokaClient();

    // should to nothing because only one of two tasks is cancelled
    cancelTasks(db::toloka::Platform::Toloka, pool(), client);

    auto txn = pool().masterWriteableTransaction();
    auto loadedTasks = db::toloka::TaskGateway{*txn}.load();
    EXPECT_THAT(
        std::vector<db::toloka::TaskStatus>({loadedTasks[0].status(), loadedTasks[1].status()}),
        ::testing::UnorderedElementsAre(
            tasks[0].status(), tasks[1].status())
    );
}

TEST_F(SuiteWithTwoTasksFixture, test_cancel_task_suite)
{
    {
        auto txn = pool().masterWriteableTransaction();
        tasks[0].setStatus(db::toloka::TaskStatus::Cancelling);
        tasks[1].setStatus(db::toloka::TaskStatus::Cancelling);
        db::toloka::TaskGateway{*txn}.updatex(tasks);
        txn->commit();
    }

    http::URL url{tolokaUrl()};
    url.setPath("/api/v1/task-suites/" + suites[0].tolokaId() + "/set-overlap-or-min");
    bool suiteCancelled = false;
    auto mockSuiteHandle = http::addMock(url, [&](const http::MockRequest& request) {
        auto value = json::Value::fromString(request.body);
        EXPECT_EQ(value["overlap"].as<int>(), 0);
        suiteCancelled = true;
        return http::MockResponse(std::string(MOCK_TASKS_SUITE_RESPONSE));
    });


    auto client = tolokaClient();

    cancelTasks(db::toloka::Platform::Toloka, pool(), client);

    EXPECT_EQ(suiteCancelled, true);

    auto txn = pool().masterWriteableTransaction();
    auto loadedTasks = db::toloka::TaskGateway{*txn}.load();
    EXPECT_EQ(loadedTasks[0].status(), db::toloka::TaskStatus::Cancelled);
    EXPECT_EQ(loadedTasks[1].status(), db::toloka::TaskStatus::Cancelled);
}

TEST_F(SuiteWithTwoTasksFixture, test_fail_cancel_task)
{
    {
        auto txn = pool().masterWriteableTransaction();
        tasks[0].setStatus(db::toloka::TaskStatus::Cancelling);
        tasks[1].setStatus(db::toloka::TaskStatus::Cancelling);
        db::toloka::TaskGateway{*txn}.updatex(tasks);
        txn->commit();
    }

    http::URL url{tolokaUrl()};
    url.setPath("/api/v1/task-suites/" + suites[0].tolokaId() + "/set-overlap-or-min");

    auto mockHandle = http::addMock(url, [&](const http::MockRequest& ) {
        return http::MockResponse::withStatus(500);
    });
    auto client = tolokaClient();

    EXPECT_THROW(cancelTasks(db::toloka::Platform::Toloka, pool(), client),
        maps::Exception);

    auto txn = pool().masterWriteableTransaction();
    auto loadedTasks = db::toloka::TaskGateway{*txn}.load();
    EXPECT_THAT(
        std::vector<db::toloka::TaskStatus>({loadedTasks[0].status(), loadedTasks[1].status()}),
        ::testing::UnorderedElementsAre(
            tasks[0].status(), tasks[1].status())
    );
}


} // maps::mrc::toloka::tests
