#include "geocode.h"

#include <yandex/maps/proto/search/address.pb.h>
#include <yandex/maps/proto/search/geocoder.pb.h>

namespace maps::mrc::ugc_uploader {

namespace {

auto convertAddress(const yandex::maps::proto::search::address::Address& addr)
    -> yandex::maps::proto::ugc_account::contributions::common::address::Address
{
    // binary compatible
    // @see
    // https://a.yandex-team.ru/arc/trunk/arcadia/maps/doc/proto/yandex/maps/proto/ugc_account/contributions/common/address.proto?rev=r9322359#L14
    auto buf = TString{};
    bool success = addr.SerializeToString(&buf);
    ASSERT(success);
    auto result = yandex::maps::proto::ugc_account::contributions::common::
        address::Address{};
    success = result.ParseFromString(buf);
    ASSERT(success);
    return result;
}

}  // namespace

auto makeGeosearchClient(const std::optional<NTvmAuth::TTvmClient>& tvmClient,
                         const http::URL& geosearchUrl)
    -> geosearch_client::Client
{
    auto config =
        geosearch_client::Configuration{geosearchUrl, "maps-core-nmaps-mrc-lt"};
    if (tvmClient) {
        config.setTvmTicketProvider(
            [&] { return tvmClient->GetServiceTicketFor("geosearch"); });
    }
    config.setRetryPolicy(common::RetryPolicy()
                              .setTryNumber(5)
                              .setInitialCooldown(std::chrono::seconds(1))
                              .setCooldownBackoff(2));
    return geosearch_client::Client{config};
}

auto search(geosearch_client::Client& geosearchClient,
            const geolib3::BoundingBox& bboxGeo,
            const std::string& lang)
    -> std::optional<yandex::maps::proto::ugc_account::contributions::common::
                         address::Address>
{
    auto searchParams = geosearch_client::SearchParams()
                            .setLang(lang)
                            .addType(geosearch_client::Type::Geo)
                            .setBboxGeo(bboxGeo);
    auto response = geosearchClient.search(searchParams);
    for (const auto& geoObject : response.reply().geo_object()) {
        for (const auto& metadata : geoObject.metadata()) {
            if (!metadata.HasExtension(yandex::maps::proto::search::geocoder::
                                           GEO_OBJECT_METADATA)) {
                continue;
            }
            auto geoObjectMetadata = metadata.GetExtension(
                yandex::maps::proto::search::geocoder::GEO_OBJECT_METADATA);
            return convertAddress(geoObjectMetadata.address());
        }
    }
    return std::nullopt;
}

}  // namespace maps::mrc::ugc_uploader
