#include "geocode.h"
#include "ugc.h"
#include "utility.h"

#include <maps/libs/http/include/client.h>
#include <maps/libs/http/include/request.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/proto/include/common.h>
#include <yandex/maps/geolib3/proto.h>
#include <yandex/maps/i18n.h>
#include <yandex/maps/proto/ugc_account/backoffice.pb.h>

namespace maps::mrc::ugc_uploader {

namespace {

auto makeModifyContributionsProto(
    const http::URL& mrcUgcBackUrl,
    const db::Ride& ride,
    const std::optional<db::Feature>& photo,
    const db::FeedbackIdToHypothesisTypeMap& hypotheses)
    -> yandex::maps::proto::ugc_account::backoffice::ModifyContributions
{
    auto result =
        yandex::maps::proto::ugc_account::backoffice::ModifyContributions{};
    auto* modifyContribution = result.add_modify_contribution();
    auto contributionId = TString{MRC_RIDE_CONTRIBUTION_NAMESPACE + ":" +
                                  std::to_string(ride.rideId())};
    if (ride.isDeleted()) {
        auto* del = modifyContribution->mutable_delete_();
        del->set_id(contributionId);
    }
    else {
        auto* upsert = modifyContribution->mutable_upsert();
        upsert->set_id(contributionId);
        auto* contributionItem = upsert->mutable_contribution();
        contributionItem->set_timestamp(
            chrono::convertToUnixTime(ride.startTime()));
        auto localizedMetadata = contributionItem->mutable_lang_to_metadata();
        for (const auto& [lang, locale] : supportedLocales()) {
            auto& contributionMetadata = (*localizedMetadata)[TString{lang}];
            auto* mrcRide = contributionMetadata.mutable_mrc_ride();
            mrcRide->set_id(TString{std::to_string(ride.rideId())});
            *mrcRide->mutable_started_at() =
                proto::makeTime(ride.startTime(), locale);
            *mrcRide->mutable_finished_at() =
                proto::makeTime(ride.endTime(), locale);
            *mrcRide->mutable_duration() =
                proto::makeDuration(ride.endTime() - ride.startTime(), locale);
            *mrcRide->mutable_distance() =
                proto::makeDistance(ride.distanceInMeters(), locale);
            mrcRide->set_photos_count(ride.photos());
            if (ride.publishedPhotos().has_value()) {
                mrcRide->set_published_photos_count(ride.publishedPhotos().value());
            }
            if (photo) {
                auto url = mrcUgcBackUrl;
                url.setPath("/v1/rides/my/photo_image")
                    .addParam("photo_id", std::to_string(photo->id()));
                *mrcRide->mutable_album_image() = proto::makeImage(
                    url, "size_name", proto::prepareNamedSizes(*photo));
            }
            auto* status = mrcRide->mutable_ride_status();
            switch (ride.status()) {
                case db::RideStatus::Pending:
                    status->mutable_pending();
                    break;
                case db::RideStatus::Processed:
                    status->mutable_processed();
                    break;
            }
            mrcRide->set_show_authorship(ride.showAuthorship());
            if (ride.clientId()) {
                mrcRide->set_client_ride_id(TString(*ride.clientId()));
            }
            db::copyRideHypotheses(hypotheses, *mrcRide);
        }
    }
    ASSERT(result.IsInitialized());
    return result;
}

auto encodeFeedbackType(db::WalkFeedbackType feedbackType)
{
    switch (feedbackType) {
        using yandex::maps::proto::ugc_account::contributions::mrc_walk_object::
            ObjectType;
        case db::WalkFeedbackType::None:
            return ObjectType::UNKNOWN_OBJECT_TYPE;
        case db::WalkFeedbackType::Barrier:
            return ObjectType::BARRIER;
        case db::WalkFeedbackType::Other:
            return ObjectType::OTHER;
        case db::WalkFeedbackType::BuildingEntrance:
            return ObjectType::BUILDING_ENTRANCE;
        case db::WalkFeedbackType::AddressPlate:
            return ObjectType::ADDRESS_PLATE;
        case db::WalkFeedbackType::EntrancePlate:
            return ObjectType::ENTRANCE_PLATE;
        case db::WalkFeedbackType::BusinessSign:
            return ObjectType::BUSINESS_SIGN;
        case db::WalkFeedbackType::BusinessWorkingHours:
            return ObjectType::BUSINESS_WORKING_HOURS;
        case db::WalkFeedbackType::FootPath:
            return ObjectType::FOOT_PATH;
        case db::WalkFeedbackType::CyclePath:
            return ObjectType::CYCLE_PATH;
        case db::WalkFeedbackType::Stairs:
            return ObjectType::STAIRS;
        case db::WalkFeedbackType::Ramp:
            return ObjectType::RAMP;
        case db::WalkFeedbackType::Room:
            return ObjectType::ROOM;
        case db::WalkFeedbackType::Wall:
            return ObjectType::WALL;
        case db::WalkFeedbackType::Organization:
            return ObjectType::ORGANIZATION;
        case db::WalkFeedbackType::Fence:
            return ObjectType::FENCE;
        case db::WalkFeedbackType::Building:
            return ObjectType::BUILDING;
        case db::WalkFeedbackType::Parking:
            return ObjectType::PARKING;
    }
}

auto encodeGeometry(const db::WalkObject& walkObject)
{
    switch (walkObject.geometryType()) {
        case geolib3::GeometryType::Point:
            return geolib3::proto::encodeGeometry(
                std::get<geolib3::Point2>(walkObject.geodeticGeometry()));
        case geolib3::GeometryType::LineString:
            return geolib3::proto::encodeGeometry(
                std::get<geolib3::Polyline2>(walkObject.geodeticGeometry()));
        case geolib3::GeometryType::Polygon:
            return geolib3::proto::encodeGeometry(
                std::get<geolib3::Polygon2>(walkObject.geodeticGeometry()));
        default:
            REQUIRE(false,
                    "unsupported geometry type " << walkObject.geometryType());
    }
}

auto encodeActionType(db::ObjectActionType actionType)
{
    switch (actionType) {
        using yandex::maps::proto::ugc_account::contributions::mrc_walk_object::
            ActionType;
        case db::ObjectActionType::Add:
            return ActionType::ADD;
        case db::ObjectActionType::Remove:
            return ActionType::REMOVE;
    }
}

auto makeShootingPoint(const geolib3::Point2& geodeticPos,
                       geolib3::Heading heading)
    -> yandex::maps::proto::common2::GeoPhoto::ShootingPoint
{
    auto result = yandex::maps::proto::common2::GeoPhoto::ShootingPoint{};

    auto point = yandex::maps::proto::common2::geometry::Point{};
    point.set_lat(geodeticPos.y());
    point.set_lon(geodeticPos.x());
    auto point3d = yandex::maps::proto::common2::GeoPhoto::Point3D{};
    *point3d.mutable_point() = point;
    // altitude is not provided
    *result.mutable_point() = point3d;

    auto direction = yandex::maps::proto::common2::geometry::Direction{};
    direction.set_azimuth(heading.value());
    direction.set_tilt(0.);  // tilt is not provided
    *result.mutable_direction() = direction;

    return result;
}

void copyStatus(const db::WalkObject& walkObject,
                yandex::maps::proto::ugc_account::contributions::
                    mrc_walk_object::ObjectContribution& result)
{
    auto* status = result.mutable_status();
    switch (walkObject.status()) {
        case db::ObjectStatus::Delayed:
        case db::ObjectStatus::Pending:
        case db::ObjectStatus::WaitPublishing:
            status->mutable_pending();
            break;
        case db::ObjectStatus::Discarded:
        case db::ObjectStatus::Failed:
            status->mutable_discarded();
            break;
        case db::ObjectStatus::Published:
            auto* published = status->mutable_published();
            if (walkObject.hasFeedbackTaskId()) {
                published->set_feedback_task_id(
                    std::to_string(walkObject.feedbackTaskId()));
            }
            break;
    }
}

void copyGeoPhotos(const db::Features& features,
                   yandex::maps::proto::ugc_account::contributions::
                       mrc_walk_object::ObjectContribution& result,
                   const http::URL& mrcUgcBackUrl)
{
    for (const auto& feature : features) {
        auto* photo = result.add_photos();
        auto url = mrcUgcBackUrl;
        url.setPath("/v1/walk_objects/my/photos")
            .addParam("photo_id", std::to_string(feature.id()));
        *photo->mutable_image() = proto::makeImage(
            url, "size_name", proto::prepareNamedSizes(feature));
        *photo->mutable_shooting_point() =
            makeShootingPoint(feature.geodeticPos(), feature.heading());
        photo->set_taken_at(chrono::convertToUnixTime(feature.timestamp()));
        // attribution is not provided
    }
}

auto makeModifyContributionsProto(const http::URL& mrcUgcBackUrl,
                                  geosearch_client::Client& geosearchClient,
                                  const db::WalkObject& walkObject,
                                  const db::Features& features)
    -> yandex::maps::proto::ugc_account::backoffice::ModifyContributions
{
    auto result =
        yandex::maps::proto::ugc_account::backoffice::ModifyContributions{};
    auto* modifyContribution = result.add_modify_contribution();
    auto contributionId = TString{MRC_WALK_OBJECT_CONTRIBUTION_NAMESPACE + ":" +
                                  std::to_string(walkObject.id())};
    auto* upsert = modifyContribution->mutable_upsert();
    upsert->set_id(contributionId);
    auto* contributionItem = upsert->mutable_contribution();
    contributionItem->set_timestamp(
        chrono::convertToUnixTime(walkObject.createdAt()));
    auto localizedMetadata = contributionItem->mutable_lang_to_metadata();
    for (const auto& [lang, locale] : supportedLocales()) {
        auto& contributionMetadata = (*localizedMetadata)[TString{lang}];
        auto* mrcWalkObject = contributionMetadata.mutable_mrc_walk_object();
        mrcWalkObject->set_id(TString{std::to_string(walkObject.id())});
        *mrcWalkObject->mutable_created_at() =
            proto::makeTime(walkObject.createdAt(), locale);
        mrcWalkObject->set_feedback_type(
            encodeFeedbackType(walkObject.feedbackType()));
        copyStatus(walkObject, *mrcWalkObject);
        *mrcWalkObject->mutable_geometry() = encodeGeometry(walkObject);
        auto bboxGeo =
            std::visit([](const auto& geom) { return geom.boundingBox(); },
                       walkObject.geodeticGeometry());
        if (auto address = search(geosearchClient, bboxGeo, lang)) {
            *mrcWalkObject->mutable_address() = *address;
        }
        if (walkObject.hasIndoorLevelId()) {
            mrcWalkObject->set_indoor_level_id(
                TString{walkObject.indoorLevelId()});
        }
        if (walkObject.hasActionType()) {
            mrcWalkObject->set_action_type(
                encodeActionType(walkObject.actionType()));
        }
        copyGeoPhotos(features, *mrcWalkObject, mrcUgcBackUrl);
        mrcWalkObject->set_comment(TString{walkObject.comment()});
        mrcWalkObject->set_client_object_id(
            std::to_string(chrono::sinceEpoch<std::chrono::milliseconds>(
                walkObject.createdAt())));
    }
    ASSERT(result.IsInitialized());
    return result;
}

auto makeDeleteContributionsProto(
    const db::WalkObject& walkObject)
    -> yandex::maps::proto::ugc_account::backoffice::ModifyContributions
{
    auto result =
        yandex::maps::proto::ugc_account::backoffice::ModifyContributions{};
    auto* modifyContribution = result.add_modify_contribution();
    auto contributionId = TString{MRC_WALK_OBJECT_CONTRIBUTION_NAMESPACE + ":" +
                                  std::to_string(walkObject.id())};
    auto* del = modifyContribution->mutable_delete_();
    del->set_id(contributionId);
    ASSERT(result.IsInitialized());
    return result;
}

void performPutRequest(
    const std::optional<NTvmAuth::TTvmClient>& tvmClient,
    const http::URL& contributionsModifyUrl,
    const std::string& userId,
    yandex::maps::proto::ugc_account::backoffice::ModifyContributions&
        modifyContributions)
{
    static auto httpClient = http::Client{};
    auto buffer = TString{};
    REQUIRE(modifyContributions.SerializeToString(&buffer),
            "failed to serialize contribution");
    auto request = http::Request(httpClient, http::PUT, contributionsModifyUrl);
    if (tvmClient) {
        request.addHeader(
            auth::SERVICE_TICKET_HEADER,
            tvmClient->GetServiceTicketFor("maps-core-ugc-backoffice"));
    }
    request.addParam("uid", userId);
    request.setContent(std::string(buffer.data(), buffer.size()));
    auto response = request.perform();
    REQUIRE(response.status() == 200 || response.status() == 201,
            "[" << request.url().toString() << "] unexpected response code: "
                << response.status() << " " << response.readBody());
}

}  // namespace

http::URL makeContributionsModifyUrl(const common::Config& config)
{
    auto result = config.externals().mapsCoreUgcBackofficeUrl();
    result.setPath("/v1/contributions/modify");
    return result;
}

void pushContribution(const std::optional<NTvmAuth::TTvmClient>& tvmClient,
                      const http::URL& contributionsModifyUrl,
                      const http::URL& mrcUgcBackUrl,
                      const db::Ride& ride,
                      const std::optional<db::Feature>& photo,
                      const db::FeedbackIdToHypothesisTypeMap& hypotheses)
{
    if (ride.userId() == "0") {
        WARN() << "ride " << ride.rideId() << " has zero user_id";
        return;
    }
    auto modifyContributions =
        makeModifyContributionsProto(mrcUgcBackUrl, ride, photo, hypotheses);
    performPutRequest(
        tvmClient, contributionsModifyUrl, ride.userId(), modifyContributions);
    INFO() << "ride " << ride.rideId() << " successfully pushed";
}

void pushContribution(const std::optional<NTvmAuth::TTvmClient>& tvmClient,
                      const http::URL& contributionsModifyUrl,
                      const http::URL& mrcUgcBackUrl,
                      const http::URL& geosearchUrl,
                      const db::WalkObject& walkObject,
                      std::future<db::Features> features)
{
    if (walkObject.dataset() != db::Dataset::Walks) {
        WARN() << "walkObject " << walkObject.id() << " has no walks dataset";
        return;
    }
    if (!walkObject.hasUserId()) {
        WARN() << "walkObject " << walkObject.id() << " has zero user_id";
        return;
    }
    auto geosearchClient = makeGeosearchClient(tvmClient, geosearchUrl);
    auto modifyContributions = makeModifyContributionsProto(
        mrcUgcBackUrl, geosearchClient, walkObject, features.get());
    performPutRequest(tvmClient,
                      contributionsModifyUrl,
                      walkObject.userId(),
                      modifyContributions);
    INFO() << "walkObject " << walkObject.id() << " successfully pushed";
}

void delWalkObjectContribution(
    const std::optional<NTvmAuth::TTvmClient>& tvmClient,
    const http::URL& contributionsModifyUrl,
    const db::WalkObject& walkObject)
{
    if (walkObject.dataset() != db::Dataset::Walks) {
        WARN() << "walkObject " << walkObject.id() << " has no walks dataset";
        return;
    }
    if (!walkObject.hasUserId()) {
        WARN() << "walkObject " << walkObject.id() << " has zero user_id";
        return;
    }
    auto modifyContributions = makeDeleteContributionsProto(walkObject);
    performPutRequest(tvmClient,
                      contributionsModifyUrl,
                      walkObject.userId(),
                      modifyContributions);
    INFO() << "walkObject " << walkObject.id() << " contribution successfully deleted";
}

const LangToLocaleMap& supportedLocales()
{
    static const auto LANG_TO_LOCALE_MAP = [] {
        auto result = LangToLocaleMap{};
        for (const auto& lang : {
                 "en_IL",
                 "en_RU",
                 "en_UA",
                 "en_US",
                 "ru_RU",
                 "ru_UA",
                 "tr_TR",
                 "uk_UA",
             }) {
            result.insert(
                {std::string{lang},
                 i18n::bestLocale(boost::lexical_cast<Locale>(lang))});
        }
        return result;
    }();
    return LANG_TO_LOCALE_MAP;
}

}  // namespace maps::mrc::ugc_uploader
