#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_utility.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/ugc_uploader/lib/db.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/ugc_uploader/lib/ugc.h>

namespace {

auto g_stop = std::atomic<bool>{};
auto g_tvmClient = std::optional<NTvmAuth::TTvmClient>{};

void setStop(int /*sig*/)
{
    g_stop.store(true);
}

}  // namespace

int main(int argc, char* argv[])
try {
    using namespace maps::mrc::ugc_uploader;
    using namespace std::chrono_literals;
    using maps::mrc::db::Ride;
    using maps::mrc::db::WalkObject;
    const auto SLEEP_DURATION = 5s;

    auto parser = maps::cmdline::Parser{};
    auto syslog = parser.string("syslog-tag")
                      .help("redirect log output to syslog with given tag");
    auto configPath = parser.string("config").help("path to configuration");
    auto secretVersion = parser.string("secret").help(
        "version for secrets from yav.yandex-team.ru");
    auto disableTvmClient = parser.flag("disable-tvm")
                                .defaultValue(false)
                                .help("disable tvm client");
    parser.parse(argc, argv);
    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }
    INFO() << "starting";
    auto config =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);
    if (!disableTvmClient) {
        g_tvmClient = maps::auth::TvmtoolSettings().makeTvmClient();
    }
    auto postgres = config.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                                          maps::mrc::common::LONG_READ_POOL_ID);
    for (int sig : {SIGTERM, SIGINT, SIGQUIT}) {
        std::signal(sig, setStop);
    }
    while (!g_stop.load()) {
        bool withoutSleep = false;
        if (queuePop<Ride>(postgres.pool(), [&](const Ride& ride) {
                pushContribution(
                    g_tvmClient,
                    makeContributionsModifyUrl(config),
                    config.externals().mapsCoreNmapsMrcUgcBackUrl(),
                    ride,
                    tryLoadSizedPhoto(
                        *postgres.pool().masterReadOnlyTransaction(), ride),
                    maps::mrc::db::loadRideHypotheses(
                        *postgres.pool().masterReadOnlyTransaction(), ride));
            })) {
            withoutSleep = true;
        }
        if (queuePop<WalkObject>(postgres.pool(), [&](const WalkObject& obj) {
                pushContribution(
                    g_tvmClient,
                    makeContributionsModifyUrl(config),
                    config.externals().mapsCoreNmapsMrcUgcBackUrl(),
                    config.externals().geosearchUrl(),
                    obj,
                    std::async(std::launch::deferred, [&] {
                        return loadPhotos(
                            *postgres.pool().masterReadOnlyTransaction(), obj);
                    }));
            })) {
            withoutSleep = true;
        }
        if (!withoutSleep) {
            std::this_thread::sleep_for(SLEEP_DURATION);
        }
    }
    INFO() << "shutting down";
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
