#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/user_activity_stat/lib/db.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

namespace maps::mrc::user_activity_stat::tests {

struct Signal {
    std::string date;
    TId user;
    bool published;
};

using Signals = std::vector<Signal>;

using namespace testing;

struct Fixture : testing::Test,
                 unittest::WithUnittestConfig<unittest::DatabaseFixture> {
};

    TEST_F(Fixture, test_db)
    {
        auto signals = Signals{
            {.date = "2020-10-01 16:55:00", .user = 1, .published = true},
            {.date = "2020-10-01 16:56:00", .user = 1, .published = true},
            {.date = "2020-10-02 10:46:00", .user = 1, .published = true},
            {.date = "2020-10-02 10:47:00", .user = 2, .published = true},
            {.date = "2020-10-02 10:48:00", .user = 2, .published = false},
        };
        auto photos = db::Features{};
        for (const auto& signal : signals) {
            auto user = boost::lexical_cast<std::string>(signal.user);
            photos
                .emplace_back(
                    sql_chemistry::GatewayAccess<db::Feature>::construct())
                .setUserId(user)
                .setSourceId(user)
                .setGeodeticPos(geolib3::Point2{0, 0})
                .setHeading(geolib3::Heading{0})
                .setTimestamp(chrono::parseSqlDateTime(signal.date))
                .setMdsKey({"1234", "abc"})
                .setDataset(db::Dataset::Rides)
                .setSize(6, 9)
                .setAutomaticShouldBePublished(signal.published)
                .setIsPublished(signal.published);
        }
        {
            auto txn = pool().masterWriteableTransaction();
            db::FeatureGateway{*txn}.insert(photos);
            txn->commit();
        }
        auto expect = TUserActivity{
            {TId(1),
             {
                 {chrono::parseSqlDateTime("2020-10-01 00:00:00"), TCounter(2)},
                 {chrono::parseSqlDateTime("2020-10-02 00:00:00"), TCounter(1)},
             }},
            {TId(2),
             {
                 {chrono::parseSqlDateTime("2020-10-02 00:00:00"), TCounter(1)},
             }},
        };
        auto userActivity = loadUserActivity(pool());
        EXPECT_EQ(expect, userActivity);
    }

}  // namespace maps::mrc::user_activity_stat::tests
