#include "feature_export.h"

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/tools/grinder/worker/include/api.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/mongo/include/init.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

namespace {

void exportFeaturesHandler(const maps::mrc::common::Config& mrcConfig,
                           const maps::grinder::worker::Task& task)
{
    const auto tasksGroupId
        = task.args()["tasks_group_id"].as<maps::mrc::db::TId>();
    const auto ytExportTablePath
        = task.args()["yt_export_table_path"].as<std::string>();

    INFO() << "Exporting features from tasks group " << tasksGroupId
           << " results";

    maps::mrc::yt_export::exportFeatures(mrcConfig, tasksGroupId,
                                         TString{ytExportTablePath});

    INFO() << "Done with the features export from tasks group "
           << tasksGroupId << " results";
}

} // anonymous namespace

int main(int argc, char* argv[]) try {
    NYT::Initialize(argc, const_cast<const char**>(argv));

    using namespace maps;
    mongo::init();

    maps::cmdline::Parser parser;
    auto configPath
        = parser.string("config").help("path to mrc configuration");
    auto syslogTag
        = parser.string("syslog-tag")
              .help("redirect log output to syslog with given tag");
    auto grinderConfigPath = parser.string("grinder-config")
                                 .help("path to grinder configuration file");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    grinder::worker::Options workerOpts;

    auto config = maps::mrc::common::templateConfigFromCmdPath(configPath);
    auto pgpoolHolder =
        config.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                              maps::mrc::common::LONG_READ_POOL_ID);

    if (grinderConfigPath.defined()) {
        workerOpts.setConfigLocation(grinderConfigPath);
    }

    workerOpts
        .on("export_features",
            [&](const grinder::worker::Task& task) {
                exportFeaturesHandler(config, task);
            })
        .setConcurrencyLevel(5);

    grinder::worker::run(workerOpts);

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const yexception& e) {
    FATAL() << "Worker failed: " << e.what();
    if (e.BackTrace()) {
        FATAL() << e.BackTrace()->PrintToString();
    }
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
