#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/walk_object.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/walk_object_gateway.h>

#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/shell_cmd.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/stringutils/include/split.h>
#include <maps/libs/sql_chemistry/include/order.h>

#include <cstdlib>
#include <iostream>

using namespace maps::mrc;

namespace {

const std::chrono::hours WARN_PENDING_HOURS{24};
const std::chrono::hours ERROR_PENDING_HOURS{72};

constexpr size_t ERROR_FAILED_OBJECTS_COUNT = 1;

size_t getObjectsCount(db::ObjectStatus status)
{
    common::Config config = common::templateConfig();
    auto poolHolder =
        config.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                              maps::mrc::common::LONG_READ_POOL_ID);
    auto txn = poolHolder.pool().slaveTransaction();

    db::WalkObjectGateway walkObjectsGateway(*txn);
    return walkObjectsGateway.count(
        db::table::WalkObject::status.equals(status));
}

std::optional<maps::chrono::TimePoint> getObjectMinUploadDate(
    db::ObjectStatus status)
{
    common::Config config = common::templateConfig();
    auto poolHolder =
        config.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                              maps::mrc::common::LONG_READ_POOL_ID);
    auto txn = poolHolder.pool().slaveTransaction();

    auto optWalkPhoto = db::FeatureGateway(*txn).tryLoadOne(
        db::table::Feature::walkObjectId == db::table::WalkObject::id &&
            db::table::WalkObject::status.equals(status),
        maps::sql_chemistry::orderBy(db::table::Feature::id)
    );

    return optWalkPhoto.has_value()
               ? std::make_optional(
                     optWalkPhoto.value().uploadedAt())
               : std::nullopt;
}

} //namespace

int main(int argc, char* argv[])
try {
    maps::log8::setLevel(maps::log8::Level::FATAL);

    maps::cmdline::Parser parser;
    auto pendingObjects = parser
        .flag("pending-objects-age")
        .help("Check the number of pending objects in the database");
    auto failedObjects = parser
        .flag("failed-objects")
        .help("Check the number of failed objects in the database");
    auto alive = parser
        .flag("alive")
        .help("Check if onfoot-processor is alive");
    parser.parse(argc, argv);

    if (pendingObjects.defined()) {
        auto optMinDate = getObjectMinUploadDate(db::ObjectStatus::Pending);

        if (!optMinDate.has_value()) {
            std::cout << "0;OK: no pending objects exist" << std::endl;
        } else {
            const auto lag = std::chrono::duration_cast<std::chrono::hours>(
                maps::chrono::TimePoint::clock::now() - optMinDate.value());

            if (lag > ERROR_PENDING_HOURS) {
                std::cout << "2;Error: the oldest pending object is " <<
                    lag.count() << " hours old" << std::endl;
            } else if (lag > WARN_PENDING_HOURS) {
                std::cout << "1;Warning: the oldest pending object is " <<
                    lag.count() << " hours old" << std::endl;
            } else {
                std::cout << "0;OK: no old pending objects exist" << std::endl;
            }
        }

        return EXIT_SUCCESS;
    }

    if (failedObjects.defined()) {
        auto count = getObjectsCount(db::ObjectStatus::Failed);

        if (count > ERROR_FAILED_OBJECTS_COUNT) {
            std::cout << "2;Error: failed objects count is " << count << std::endl;
        } else {
            std::cout << "0;OK: failed objects count is " << count << std::endl;
        }
        return EXIT_SUCCESS;
    }

    if (alive.defined()) {
        auto result = maps::shell::runCmd("supervisorctl status onfoot-processor");
        if (result.exitCode != 0) {
            std::cout << "2;Error: onfoot-processor is probably dead: failed to run supervisorctl" << std::endl;
        } else {
            //Expected result: "onfoot-processor          RUNNING    pid 262, uptime 49 days, 0:57:57"

            auto fields = maps::stringutils::fields(result.stdOut);
            if (fields.size() < 2 || fields[1] != "RUNNING") {
                std::cout << "2;Error: onfoot-processor is probably dead: '" << result.stdOut << "'" << std::endl;
            } else {
                std::cout << "0;OK: onfoot-processor is alive " << std::endl;
            }
        }
    }

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    std::cout << "2;Error: " << e.what() << std::endl;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    std::cout << "2;Error: " << e.what() << std::endl;
    return EXIT_FAILURE;
}
