#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/http/include/test_utils.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/walk_object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/onfoot-processor/lib/worker.h>
#include <util/system/env.h>
#include <maps/libs/geolib/include/polyline.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>
#include <yandex/maps/wiki/unittest/json_schema.h>

using namespace testing;

namespace maps::mrc::onfoot::tests {

namespace {

const std::string TEST_UID = "1";
const std::string DEVICE_ID = "MyFluffyPhone";
const std::string COMMENT = "Hello, world!";
const std::string INDOOR_LEVEL = "1";
const std::string NMAPS_OBJECT_ID = "123";
const std::string MDS_KEY = "4436";
const std::string MDS_PATH = "development/walk/images/17673352";

constexpr db::TId FEEDBACK_TASK_ID = 12345;

const std::string SOCIAL_URL = "http://localhost/feedback/tasks/onfoot";

const std::string SOCIAL_FEEDBACK_SCHEMA_PATH =
    SRC_("../../../../schemas/social/social.post.onfoot_feedback_task.request.schema.json");

enum class Publish { Yes, No };

class Fixture : public unittest::WithUnittestConfig<unittest::DatabaseFixture>
{
public:
    static Fixture& instance()
    {
        static Fixture fixture;
        return fixture;
    }

    db::WalkObject prepareObject(db::ObjectStatus status)
    {
        auto txn = pool().masterWriteableTransaction();

        db::WalkObject walkObject(
            db::Dataset::Walks,
            DEVICE_ID,
            chrono::TimePoint::clock::now(),
            db::WalkFeedbackType::Other,
            geolib3::Point2(0.0, 0.1),
            COMMENT
        );
        walkObject.setObjectStatus(status);

        db::WalkObjectGateway walkObjectsGateway(*txn);
        walkObjectsGateway.insertx(walkObject);
        txn->commit();
        return walkObject;
    }

    void addWalkPhoto(
        const db::WalkObject& walkObject,
        std::optional<Publish> publish = std::nullopt,
        db::FeaturePrivacy privacy = db::FeaturePrivacy::Public)
    {
        auto txn = pool().masterWriteableTransaction();
        auto feature = sql_chemistry::GatewayAccess<db::Feature>::construct()
            .setGeodeticPos(geolib3::Point2(0.0, 0.0))
            .setHeading(geolib3::Heading(0.0))
            .setTimestamp(maps::chrono::parseIsoDate("2021-06-04 16:35:15"))
            .setMdsKey({MDS_KEY, MDS_PATH})
            .setSize(64, 64)
            .setUserId(TEST_UID)
            .setWalkObjectId(walkObject.id())
            .setPrivacy(privacy);
        if (publish.has_value()) {
            feature
                .setAutomaticShouldBePublished(publish.value() == Publish::Yes)
                .setProcessedAt(maps::chrono::parseIsoDate("2021-06-04 16:54:57"));
        }
        db::FeatureGateway{*txn}.insert(feature);

        txn->commit();
    }

    db::WalkObject prepareObjectAndPhoto(
        db::ObjectStatus status,
        std::optional<Publish> publish = std::nullopt,
        db::FeaturePrivacy privacy = db::FeaturePrivacy::Public)
    {
        auto walkObject = prepareObject(status);
        addWalkPhoto(walkObject, publish, privacy);
        return walkObject;
    }
};

auto createObjectAndProcessIt(std::optional<Publish> publish = std::nullopt)
{
    auto& fixture = Fixture::instance();
    auto txn = fixture.pool().masterWriteableTransaction();

    auto walkObject =
        fixture.prepareObjectAndPhoto(db::ObjectStatus::Pending, publish);
    bool hasObjects =
        Worker(fixture.config(), fixture.pool(), DryRun::No).runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    auto walkPhoto = db::FeatureGateway(*txn)
        .loadOne(db::table::Feature::walkObjectId == walkObject.id());

    return std::make_pair(walkObject, walkPhoto);
}

size_t featureTransactionCount()
{
    auto txn = Fixture::instance().pool().masterReadOnlyTransaction();
    return db::FeatureTransactionGateway{*txn}.count();
}

struct ExpectedSocialParams
{
    size_t numberOfPhotos = 1;
    bool isHidden = false;
    std::optional<bool> internalContent = std::nullopt;
    std::optional<geolib3::Point2> position = std::nullopt;
    std::optional<std::string> indoorLevel = std::nullopt;
    std::optional<std::string> source = std::nullopt;
    std::optional<db::GeometryVariant> geometryAfter = std::nullopt;
    std::optional<db::GeometryVariant> geometryBefore = std::nullopt;
    std::optional<std::string> objectId = std::nullopt;
};

void checkGeometry(
    const db::GeometryVariant& geometryVariant,
    const json::Value& geojson)
{
    if (auto geom = std::get_if<geolib3::Point2>(&geometryVariant)) {
        auto point = geolib3::readGeojson<geolib3::Point2>(geojson);
        EXPECT_THAT(point, *geom);
    } else if (auto geom = std::get_if<geolib3::Polyline2>(&geometryVariant)) {
        auto polyline = geolib3::readGeojson<geolib3::Polyline2>(geojson);
        EXPECT_THAT(polyline, *geom);
    } else if (auto geom =  std::get_if<geolib3::Polygon2>(&geometryVariant)) {
        auto polygon = geolib3::readGeojson<geolib3::Polygon2>(geojson);
        EXPECT_THAT(polygon.exteriorRing().toPolyline(), geom->exteriorRing().toPolyline());
    } else {
       EXPECT_FALSE("Invalid objectDiff geometry");
    }
}

auto addSocialMock(const ExpectedSocialParams& expected = {})
{
    return http::addMock(SOCIAL_URL, [=](const http::MockRequest& request) {
        wiki::unittest::validateJson(request.body, SOCIAL_FEEDBACK_SCHEMA_PATH);

        auto requestJson = json::Value::fromString(request.body);
        EXPECT_THAT(requestJson["hidden"].as<bool>(), expected.isHidden);
        if (expected.internalContent.has_value()) {
            EXPECT_THAT(requestJson["internalContent"].as<bool>(),
                expected.internalContent.value());
        }
        EXPECT_THAT(requestJson["userComment"].as<std::string>(), COMMENT);
        EXPECT_TRUE(requestJson.hasField("type"));
        EXPECT_TRUE(requestJson.hasField("source"));
        if (expected.source.has_value()) {
            EXPECT_EQ(requestJson["source"].as<std::string>(),
                expected.source.value());
        }

        if (expected.numberOfPhotos > 0) {
            auto& photos = requestJson["sourceContext"]["content"]["imageFeatures"];
            EXPECT_THAT(photos.size(), expected.numberOfPhotos);
            for (size_t i = 0; i < photos.size(); ++i) {
                EXPECT_TRUE(photos[i]["targetGeometry"].exists());
            }
        }

        if (expected.position) {
            auto pos = geolib3::readGeojson<geolib3::Point2>(requestJson["position"]);
            EXPECT_THAT(pos, *expected.position);
        }

        if (expected.geometryAfter || expected.geometryBefore) {
            auto geometryDiff = requestJson["objectDiff"]["modified"]["geometry"];
            if (expected.geometryAfter) {
                EXPECT_EQ(geometryDiff["after"].size(), 1u);
                checkGeometry(*expected.geometryAfter, geometryDiff["after"][0]);
            } else {
                EXPECT_TRUE(geometryDiff["after"].empty());
            }
            if (expected.geometryBefore) {
                EXPECT_EQ(geometryDiff["before"].size(), 1u);
                checkGeometry(*expected.geometryBefore, geometryDiff["before"][0]);
            } else {
                EXPECT_TRUE(geometryDiff["before"].empty());
            }
        } else {
            EXPECT_FALSE(requestJson.hasField("objectDiff"));
        }

        json::Builder builder;
        builder << [&](json::ObjectBuilder builder) {
            builder["feedbackTask"] << [&](json::ObjectBuilder builder) {
                builder["id"] << FEEDBACK_TASK_ID;
            };
        };
        if (expected.objectId.has_value()) {
            EXPECT_EQ(requestJson["objectId"].as<std::string>(),
                      expected.objectId.value());
        }
        else {
            EXPECT_FALSE(requestJson["objectId"].exists());
        }
        return http::MockResponse(builder.str());
    });
}

} // namespace

TEST(onfoot_processor, test_no_pending_objects)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();

    Worker worker(fixture.config(), fixture.pool(), DryRun::No);

    auto txn = fixture.pool().masterWriteableTransaction();

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // empty database

    fixture.prepareObjectAndPhoto(db::ObjectStatus::Discarded, Publish::Yes);
    fixture.prepareObjectAndPhoto(db::ObjectStatus::Failed, Publish::Yes);
    fixture.prepareObjectAndPhoto(db::ObjectStatus::Published, Publish::Yes);

    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no objects in the pending state

    db::WalkObjectGateway walkObjectsGateway(*txn);
    auto walkObjects = walkObjectsGateway.load(
        db::table::WalkObject::status.equals(db::ObjectStatus::Pending));
    EXPECT_TRUE(walkObjects.empty()); // still no objects in the pending state
}

TEST(onfoot_processor, test_social_temporal_error)
{
    Fixture::instance().postgres().truncateTables();

    size_t socialRequestCount = 0;

    auto socialMock = http::addMock(
        SOCIAL_URL,
        [&](const http::MockRequest& request) {
            socialRequestCount++;
            if (socialRequestCount == 1) {
                return http::MockResponse(R"(
                {
                    "error": {
                        "status": "500",
                        "message": "Something wrong happend"
                    }
                })");
            } else {
                wiki::unittest::validateJson(request.body, SOCIAL_FEEDBACK_SCHEMA_PATH);

                auto requestJson = json::Value::fromString(request.body);
                EXPECT_THAT(requestJson["hidden"].as<bool>(), false);
                EXPECT_THAT(requestJson["userComment"].as<std::string>(), COMMENT);
                EXPECT_TRUE(requestJson["sourceContext"]["content"]["imageFeatures"][0]
                                    ["targetGeometry"]
                                        .exists());

                json::Builder builder;
                builder << [&](json::ObjectBuilder builder) {
                    builder["feedbackTask"] << [&](json::ObjectBuilder builder) {
                        builder["id"] << FEEDBACK_TASK_ID;
                    };
                };

                return http::MockResponse(builder.str());
            }
        });

    auto& fixture = Fixture::instance();
    auto txn = fixture.pool().masterWriteableTransaction();

    auto walkObject = fixture.prepareObjectAndPhoto(db::ObjectStatus::Pending, Publish::Yes);
    auto worker = Worker(fixture.config(), fixture.pool(), DryRun::No);
    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    auto walkPhoto = db::FeatureGateway(*txn)
        .loadOne(db::table::Feature::walkObjectId == walkObject.id());

    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_EQ(walkPhoto.shouldBePublished(), true);
    {
        walkPhoto.setIsPublished(true);
        auto txnUpdate = fixture.pool().masterWriteableTransaction();
        db::FeatureGateway(*txnUpdate).update(walkPhoto, db::UpdateFeatureTxn::No);
        txnUpdate->commit();
    }

    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_EQ(walkObject.status(), db::ObjectStatus::WaitPublishing);

    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());

    txn = fixture.pool().masterWriteableTransaction();
    hasObjects =
        Worker(fixture.config(), fixture.pool(), DryRun::No).runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no more pending objects left
}

TEST(onfoot_processor, test_forbidden_content)
{
    Fixture::instance().postgres().truncateTables();

    const auto [walkObject, walkPhoto] = createObjectAndProcessIt(Publish::No);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Discarded);
    EXPECT_THAT(walkPhoto.isPublished(), false);
    EXPECT_THAT(walkPhoto.shouldBePublished(), false);
    EXPECT_TRUE(walkPhoto.processedAt().has_value());
    EXPECT_THAT(featureTransactionCount(), 0u);
}

TEST(onfoot_processor, test_forbidden_not_defined)
{
    Fixture::instance().postgres().truncateTables();

    const auto [walkObject, walkPhoto] = createObjectAndProcessIt();
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Pending);
    EXPECT_THAT(walkPhoto.isPublished(), false);
    EXPECT_THAT(walkPhoto.shouldBePublished(), false);
    EXPECT_FALSE(walkPhoto.processedAt().has_value());
    EXPECT_THAT(featureTransactionCount(), 0u);
}

TEST(onfoot_processor, test_hide_photos_in_unallowed_regions)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();

    auto socialMock = http::addMock(
        SOCIAL_URL,
        [&](const http::MockRequest& request) {
            EXPECT_THAT(json::Value::fromString(request.body)["hidden"].as<bool>(), true);
            return http::MockResponse(R"({"feedbackTask":{"id":12345}})");
        });

    auto txn = fixture.pool().masterWriteableTransaction();
    auto walkObject = fixture.prepareObjectAndPhoto(
        db::ObjectStatus::Pending, Publish::Yes, db::FeaturePrivacy::Secret);
    Worker(fixture.config(), fixture.pool(), DryRun::No).runOnce(*txn);
    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_THAT(featureTransactionCount(), 0u);

    auto walkPhoto = db::FeatureGateway(*txn)
        .loadOne(db::table::Feature::walkObjectId == walkObject.id());
    EXPECT_THAT(walkPhoto.hasPrivacy(), true);
    EXPECT_THAT(walkPhoto.privacy(), db::FeaturePrivacy::Secret);
}

TEST(onfoot_processor, test_success)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();

    auto socialMock = addSocialMock();

    auto [walkObject, walkPhoto] = createObjectAndProcessIt(Publish::Yes);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_FALSE(walkObject.hasFeedbackTaskId());
    EXPECT_FALSE(walkPhoto.isPublished());
    EXPECT_TRUE(walkPhoto.shouldBePublished());
    EXPECT_THAT(walkPhoto.hasPrivacy(), true);
    EXPECT_THAT(walkPhoto.privacy(), db::FeaturePrivacy::Public);
    EXPECT_THAT(walkPhoto.size().width, 64ul);
    EXPECT_THAT(walkPhoto.size().height, 64ul);
    EXPECT_THAT(featureTransactionCount(), 0u);

    auto txn = fixture.pool().masterWriteableTransaction();
    const auto walkObjects = db::WalkObjectGateway(*txn).load(
        db::table::WalkObject::status.equals(db::ObjectStatus::Pending));
    EXPECT_TRUE(walkObjects.empty()); // no objects in the pending state

    // Mark feature published and run the worker again
    walkPhoto.setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhoto, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = fixture.pool().masterWriteableTransaction();
    auto hasObjects =
        Worker(fixture.config(), fixture.pool(), DryRun::No).runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no more pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());

    txn = fixture.pool().masterWriteableTransaction();
    hasObjects =
        Worker(fixture.config(), fixture.pool(), DryRun::No).runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no more pending objects left

}

TEST(onfoot_processor, test_two_photos_success)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    auto socialMock = addSocialMock(ExpectedSocialParams{.numberOfPhotos = 2});

    auto walkObject = fixture.prepareObject(db::ObjectStatus::Pending);
    fixture.addWalkPhoto(walkObject, Publish::Yes);
    fixture.addWalkPhoto(walkObject, Publish::Yes);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_FALSE(walkObject.hasFeedbackTaskId());

    auto walkPhotos = db::FeatureGateway(*txn)
        .load(db::table::Feature::walkObjectId == walkObject.id());
    EXPECT_THAT(walkPhotos.size(), 2u);
    for (const auto& photo : walkPhotos) {
        EXPECT_TRUE(photo.shouldBePublished());
        EXPECT_FALSE(photo.isPublished());
    }
    EXPECT_THAT(featureTransactionCount(), 0u);

    // Mark photos published and run the worker again
    txn = pgPool.masterWriteableTransaction();
    for (auto& photo : walkPhotos) {
        photo.setIsPublished(true);
    }
    db::FeatureGateway(*txn).update(walkPhotos, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no more pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}

TEST(onfoot_processor, test_two_photos_published_separately)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    auto socialMock = addSocialMock(ExpectedSocialParams{.numberOfPhotos = 2});

    auto walkObject = fixture.prepareObject(db::ObjectStatus::Pending);
    fixture.addWalkPhoto(walkObject, Publish::Yes);
    fixture.addWalkPhoto(walkObject, Publish::Yes);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_FALSE(walkObject.hasFeedbackTaskId());

    auto walkPhotos = db::FeatureGateway(*txn)
        .load(db::table::Feature::walkObjectId == walkObject.id());
    ASSERT_THAT(walkPhotos.size(), 2u);

    // Mark only the first photo published and run the worker again
    txn = pgPool.masterWriteableTransaction();
    walkPhotos[0].setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhotos, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_FALSE(walkObject.hasFeedbackTaskId());

    // Mark second photo published and run the worker again
    txn = pgPool.masterWriteableTransaction();
    walkPhotos[1].setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhotos, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}

TEST(onfoot_processor, test_two_photos_one_forbidden)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    auto socialMock = addSocialMock();

    auto walkObject = fixture.prepareObject(db::ObjectStatus::Pending);
    fixture.addWalkPhoto(walkObject, Publish::Yes);
    fixture.addWalkPhoto(walkObject, Publish::No);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_FALSE(walkObject.hasFeedbackTaskId());

    auto walkPhotos = db::FeatureGateway(*txn)
        .load(db::table::Feature::walkObjectId == walkObject.id());
    ASSERT_THAT(walkPhotos.size(), 2u);
    EXPECT_TRUE(walkPhotos[0].shouldBePublished());
    EXPECT_FALSE(walkPhotos[1].shouldBePublished());

    // Mark first photo published
    txn = pgPool.masterWriteableTransaction();
    walkPhotos[0].setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhotos, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}

TEST(onfoot_processor, test_two_photos_one_hidden)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    auto socialMock = addSocialMock(
        ExpectedSocialParams{.numberOfPhotos = 2,
                             .isHidden = true});

    auto walkObject = fixture.prepareObject(db::ObjectStatus::Pending);
    fixture.addWalkPhoto(walkObject, Publish::Yes, db::FeaturePrivacy::Public);
    fixture.addWalkPhoto(walkObject, Publish::Yes, db::FeaturePrivacy::Restricted);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_FALSE(walkObject.hasFeedbackTaskId());

    auto walkPhotos = db::FeatureGateway(*txn)
        .load(db::table::Feature::walkObjectId == walkObject.id());
    ASSERT_THAT(walkPhotos.size(), 2u);
    EXPECT_TRUE(walkPhotos[0].shouldBePublished());
    EXPECT_TRUE(walkPhotos[1].shouldBePublished());

    // Mark photos published
    txn = pgPool.masterWriteableTransaction();
    walkPhotos[0].setIsPublished(true);
    walkPhotos[1].setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhotos, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}

TEST(onfoot_processor, test_two_photos_one_secret)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    auto socialMock = addSocialMock(
        ExpectedSocialParams{.numberOfPhotos = 2,
                             .isHidden = true,
                             .internalContent = true});

    auto walkObject = fixture.prepareObject(db::ObjectStatus::Pending);
    fixture.addWalkPhoto(walkObject, Publish::Yes, db::FeaturePrivacy::Public);
    fixture.addWalkPhoto(walkObject, Publish::Yes, db::FeaturePrivacy::Secret);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_FALSE(walkObject.hasFeedbackTaskId());

    auto walkPhotos = db::FeatureGateway(*txn)
        .load(db::table::Feature::walkObjectId == walkObject.id());
    ASSERT_THAT(walkPhotos.size(), 2u);
    EXPECT_TRUE(walkPhotos[0].shouldBePublished());
    EXPECT_TRUE(walkPhotos[1].shouldBePublished());

    // Mark photos published
    txn = pgPool.masterWriteableTransaction();
    walkPhotos[0].setIsPublished(true);
    walkPhotos[1].setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhotos, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}

TEST(onfoot_processor, test_publish_secret_photos)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    auto socialMock = addSocialMock(
        ExpectedSocialParams{.numberOfPhotos = 1,
                             .isHidden = true,
                             .internalContent = true});

    auto walkObject = fixture.prepareObject(db::ObjectStatus::Pending);
    fixture.addWalkPhoto(walkObject, Publish::Yes, db::FeaturePrivacy::Secret);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::WaitPublishing);
    EXPECT_FALSE(walkObject.hasFeedbackTaskId());

    auto walkPhotos = db::FeatureGateway(*txn)
        .load(db::table::Feature::walkObjectId == walkObject.id());
    ASSERT_THAT(walkPhotos.size(), 1u);
    EXPECT_TRUE(walkPhotos[0].shouldBePublished());
    EXPECT_TRUE(walkPhotos[0].isPublished());

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}

TEST(onfoot_processor, test_object_with_polyline_geometry)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    const geolib3::Polyline2 polyline{geolib3::PointsVector{{0.0, 0.1}, {0.0, 0.2}, {0.1, 0.0}}};

    auto socialMock = addSocialMock(
        ExpectedSocialParams{
            .isHidden = true,
            .position = geolib3::Point2{0.0333333, 0.1},
            .geometryAfter = polyline
        });

    auto walkObject = [&]{
        auto txn = pgPool.masterWriteableTransaction();

        db::WalkObject walkObject(
            db::Dataset::PedestrianTask,
            DEVICE_ID,
            chrono::TimePoint::clock::now(),
            db::WalkFeedbackType::Other,
            polyline,
            COMMENT
        );
        walkObject.setObjectStatus(db::ObjectStatus::Pending);

        db::WalkObjectGateway walkObjectsGateway(*txn);
        walkObjectsGateway.insertx(walkObject);
        txn->commit();
        return walkObject;
    }();

    fixture.addWalkPhoto(walkObject, Publish::Yes);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    // Mark photos published and run the worker again
    txn = pgPool.masterWriteableTransaction();
    auto walkPhoto = db::FeatureGateway(*txn).loadOne(
        db::table::Feature::walkObjectId == walkObject.id());
    walkPhoto.setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhoto, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no more pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}

TEST(onfoot_processor, test_object_with_no_photo)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    const geolib3::Polyline2 polyline{geolib3::PointsVector{{0.0, 0.1}, {0.0, 0.2}, {0.1, 0.0}}};

    auto socialMock = addSocialMock(
        ExpectedSocialParams{
            .numberOfPhotos = 0,
            .isHidden = true,
            .position = geolib3::Point2{0.0333333, 0.1},
            .geometryAfter = polyline
        });

    auto walkObject = [&]{
        auto txn = pgPool.masterWriteableTransaction();

        db::WalkObject walkObject(
            db::Dataset::PedestrianTask,
            DEVICE_ID,
            chrono::TimePoint::clock::now(),
            db::WalkFeedbackType::Other,
            polyline,
            COMMENT
        );
        walkObject.setObjectStatus(db::ObjectStatus::Pending);

        db::WalkObjectGateway walkObjectsGateway(*txn);
        walkObjectsGateway.insertx(walkObject);
        txn->commit();
        return walkObject;
    }();

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no more pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}


TEST(onfoot_processor, test_object_with_polygon_geometry_and_indoor_level)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    auto polygon = geolib3::Polygon2(
        geolib3::PointsVector{{0.0, 0.0}, {0.0, 0.2}, {0.2, 0.2}, {0.2, 0.0}});

    auto socialMock = addSocialMock(
        ExpectedSocialParams{
            .isHidden = true,
            .position = geolib3::Point2{0.1, 0.1},
            .indoorLevel = INDOOR_LEVEL,
            .source = "partner-pedestrian-indoor",
            .geometryAfter = polygon,
            .objectId = NMAPS_OBJECT_ID,
            });

    auto walkObject = [&]{
        auto txn = pgPool.masterWriteableTransaction();

        db::WalkObject walkObject(
            db::Dataset::PedestrianTask,
            DEVICE_ID,
            chrono::TimePoint::clock::now(),
            db::WalkFeedbackType::Other,
            polygon,
            COMMENT
        );
        walkObject.setObjectStatus(db::ObjectStatus::Pending)
                  .setIndoorLevelId(INDOOR_LEVEL)
                  .setNmapsObjectId(NMAPS_OBJECT_ID)
                  ;

        db::WalkObjectGateway walkObjectsGateway(*txn);
        walkObjectsGateway.insertx(walkObject);
        txn->commit();
        return walkObject;
    }();

    fixture.addWalkPhoto(walkObject, Publish::Yes);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    // Mark photos published and run the worker again
    txn = pgPool.masterWriteableTransaction();
    auto walkPhoto = db::FeatureGateway(*txn).loadOne(
        db::table::Feature::walkObjectId == walkObject.id());
    walkPhoto.setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhoto, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no more pending objects

    db::WalkObjectGateway(*txn).reload(walkObject);
    EXPECT_THAT(walkObject.status(), db::ObjectStatus::Published);
    EXPECT_TRUE(walkObject.hasFeedbackTaskId());
}

TEST(onfoot_processor, test_object_action_type)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();

    const geolib3::Polyline2 polyline{geolib3::PointsVector{{0.0, 0.1}, {0.0, 0.2}, {0.1, 0.0}}};

    auto makeWalkObject = [&](db::ObjectActionType actionType){
        auto txn = pgPool.masterWriteableTransaction();

        db::WalkObject walkObject(
            db::Dataset::PedestrianTask,
            DEVICE_ID,
            chrono::TimePoint::clock::now(),
            db::WalkFeedbackType::Other,
            polyline,
            COMMENT
        );
        walkObject.setObjectStatus(db::ObjectStatus::Pending);
        walkObject.setActionType(actionType);

        db::WalkObjectGateway walkObjectsGateway(*txn);
        walkObjectsGateway.insertx(walkObject);
        txn->commit();
        return walkObject;
    };

    for (auto actionType : {db::ObjectActionType::Add,
                            db::ObjectActionType::Remove})
    {
        auto socialParams = ExpectedSocialParams{.isHidden = true, .position = geolib3::Point2{0.0333333, 0.1}};
        switch(actionType) {
            case db::ObjectActionType::Add:
                socialParams.geometryAfter = polyline;
                break;
            case db::ObjectActionType::Remove:
                socialParams.geometryBefore = polyline;
                break;
        }
        auto socialMock = addSocialMock(socialParams);

        auto txn = pgPool.masterWriteableTransaction();
        auto walkObject = makeWalkObject(actionType);
        fixture.addWalkPhoto(walkObject, Publish::Yes);
        auto worker = Worker(fixture.config(), pgPool, DryRun::No);
        bool hasObjects = worker.runOnce(*txn);
        EXPECT_TRUE(hasObjects);

        // Mark photo published
        txn = pgPool.masterWriteableTransaction();
        auto walkPhoto = db::FeatureGateway(*txn).loadOne(
            db::table::Feature::walkObjectId == walkObject.id());
        walkPhoto.setIsPublished(true);
        db::FeatureGateway(*txn).update(walkPhoto, db::UpdateFeatureTxn::No);
        txn->commit();

        txn = pgPool.masterWriteableTransaction();
        hasObjects = worker.runOnce(*txn);
        EXPECT_FALSE(hasObjects);

        db::WalkObjectGateway(*txn).reload(walkObject);
        EXPECT_EQ(walkObject.status(), db::ObjectStatus::Published);
    }
}

TEST(onfoot_processor, test_set_comment_from_object_type)
{
    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();
    auto& pgPool = fixture.pool();
    auto txn = pgPool.masterWriteableTransaction();

    const std::string AUTO_COMMENT = "Лестница/Пандус";

    auto socialMock = http::addMock(SOCIAL_URL, [=](const http::MockRequest& request) {
        wiki::unittest::validateJson(request.body, SOCIAL_FEEDBACK_SCHEMA_PATH);
        auto requestJson = json::Value::fromString(request.body);
        EXPECT_THAT(requestJson["userComment"].as<std::string>(), AUTO_COMMENT);

        json::Builder builder;
        builder << [&](json::ObjectBuilder builder) {
            builder["feedbackTask"] << [&](json::ObjectBuilder builder) {
                builder["id"] << FEEDBACK_TASK_ID;
            };
        };
        return http::MockResponse(builder.str());
    });

    auto walkObject = [&](){
        auto txn = pgPool.masterWriteableTransaction();

        db::WalkObject walkObject(
            db::Dataset::PedestrianTask,
            DEVICE_ID,
            chrono::TimePoint::clock::now(),
            db::WalkFeedbackType::Stairs,
            geolib3::Point2(0.0, 0.1),
            std::string{} // empty comment
        );
        walkObject.setObjectStatus(db::ObjectStatus::Pending);

        db::WalkObjectGateway walkObjectsGateway(*txn);
        walkObjectsGateway.insertx(walkObject);
        txn->commit();
        return walkObject;
    }();

    fixture.addWalkPhoto(walkObject, Publish::Yes);

    auto worker = Worker(fixture.config(), pgPool, DryRun::No);

    bool hasObjects = worker.runOnce(*txn);
    EXPECT_TRUE(hasObjects);

    // Mark photos published and run the worker again
    txn = pgPool.masterWriteableTransaction();
    auto walkPhoto = db::FeatureGateway(*txn).loadOne(
        db::table::Feature::walkObjectId == walkObject.id());
    walkPhoto.setIsPublished(true);
    db::FeatureGateway(*txn).update(walkPhoto, db::UpdateFeatureTxn::No);
    txn->commit();

    txn = pgPool.masterWriteableTransaction();
    hasObjects = worker.runOnce(*txn);
    EXPECT_FALSE(hasObjects); // no more pending objects
}

TEST(onfoot_processor, test_group_hypothesis_generation)
{
    const auto OBJECTS_NUMBER = 2u;
    const auto PEDESTRIAN_TASK_ID = std::string{"42"};

    auto& fixture = Fixture::instance();
    fixture.postgres().truncateTables();

    auto walkObjects = db::WalkObjects{};
    for (size_t i = 0; i < OBJECTS_NUMBER; ++i) {
        walkObjects
            .emplace_back(db::Dataset::PedestrianTask,
                          DEVICE_ID,
                          chrono::TimePoint::clock::now(),
                          db::WalkFeedbackType::Wall,
                          geolib3::Polyline2{geolib3::PointsVector{
                              {44 + 0.1 * i, 56 + 0.1 * i},
                              {44 + 0.1 * (i + 1), 56 + 0.1 * (i + 1)}}},
                          "comment" + std::to_string(i))
            .setUserId(TEST_UID)
            .setTaskId(PEDESTRIAN_TASK_ID)
            .setActionType(db::ObjectActionType::Add)
            .setObjectStatus(db::ObjectStatus::Pending);
    }
    {
        auto txn = fixture.pool().masterWriteableTransaction();
        db::WalkObjectGateway{*txn}.insertx(walkObjects);
        txn->commit();
    }
    for (const auto& walkObject : walkObjects) {
        fixture.addWalkPhoto(walkObject, Publish::Yes);
    }

    auto socialRequestCount = 0u;
    auto socialMock =
        http::addMock(SOCIAL_URL, [&](const http::MockRequest& request) {
            if (++socialRequestCount != 1u) {
                ASSERT(false);
                return http::MockResponse(R"(
                {
                    "error": {
                        "status": "500",
                        "message": "Something wrong happend"
                    }
                })");
            }

            wiki::unittest::validateJson(request.body,
                                         SOCIAL_FEEDBACK_SCHEMA_PATH);
            auto requestJson = json::Value::fromString(request.body);
            auto photos =
                requestJson["sourceContext"]["content"]["imageFeatures"];
            EXPECT_EQ(photos.size(), walkObjects.size());
            auto geometryDiffAfter =
                requestJson["objectDiff"]["modified"]["geometry"]["after"];
            EXPECT_EQ(geometryDiffAfter.size(), walkObjects.size());
            auto userComment = requestJson["userComment"].as<std::string>();
            for (size_t i = 0; i < OBJECTS_NUMBER; ++i) {
                checkGeometry(walkObjects[i].geodeticGeometry(),
                              geometryDiffAfter[i]);
                EXPECT_NE(userComment.find(walkObjects[i].comment()),
                          std::string::npos);
            }

            auto builder = json::Builder{};
            builder << [&](json::ObjectBuilder builder) {
                builder["feedbackTask"] << [&](json::ObjectBuilder builder) {
                    builder["id"] << FEEDBACK_TASK_ID;
                };
            };
            return http::MockResponse(builder.str());
        });

    auto worker = Worker(fixture.config(), fixture.pool(), DryRun::No);

    worker.runOnce(*fixture.pool().masterWriteableTransaction());

    for (auto& walkObject : walkObjects) {
        db::WalkObjectGateway{*fixture.pool().masterReadOnlyTransaction()}
            .reload(walkObject);
        EXPECT_EQ(walkObject.status(), db::ObjectStatus::WaitPublishing);
    }

    {
        auto txn = fixture.pool().masterWriteableTransaction();
        auto walkPhotos = db::FeatureGateway(*txn).load();
        for (auto& walkPhoto : walkPhotos) {
            walkPhoto.setIsPublished(true);
        }
        db::FeatureGateway{*txn}.update(walkPhotos, db::UpdateFeatureTxn::No);
        txn->commit();
    }

    worker.runOnce(*fixture.pool().masterWriteableTransaction());

    for (auto& walkObject : walkObjects) {
        db::WalkObjectGateway{*fixture.pool().masterReadOnlyTransaction()}
            .reload(walkObject);
        EXPECT_EQ(walkObject.status(), db::ObjectStatus::Published);
        EXPECT_EQ(walkObject.feedbackTaskId(), FEEDBACK_TASK_ID);
    }
}

} // namespace maps::mrc::onfoot::tests
