#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>

#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/sql_chemistry/include/gateway.h>
#include <yandex/maps/i18n.h>
#include <yandex/maps/i18n/units.h>
#include <maps/libs/deprecated/localeutils/include/locale.h>
#include <maps/libs/stringutils/include/split.h>
#include <maps/infra/yacare/include/yacare.h>

#include <boost/lexical_cast.hpp>

#include <type_traits>
#include <stdio.h>
#include <vector>
#include <unordered_map>
#include <optional>



namespace maps {
namespace mrc {
namespace tasks_planner {

struct TasksGroupKit {
    db::ugc::TasksGroup obj;
    db::ugc::TasksGroupEmails emails;
};

using TasksGroupEmailsMap =
    std::unordered_map<db::TId, db::ugc::TasksGroupEmails>;

using TaskStatusesMap =
    std::unordered_map<db::TId,
                        std::unordered_map<db::ugc::TaskStatus, size_t>>;

using TaskAssignmentMap = std::unordered_map<db::TId, db::ugc::Assignment>;

using AssignmentReviewsMap = std::unordered_map<db::TId, db::ugc::AssignmentReviews>;

const Locale RU = boost::lexical_cast<Locale>("ru");

void setTotalCountHeader(yacare::Response& response, size_t count);

template <typename T>
std::vector<T> vectorQueryParam(
    const yacare::Request& request,
    const std::string& name)
{
    std::vector<T> retVal;

    auto commaSeparatedList = request.input()[name];
    if (commaSeparatedList.empty()) {
        return retVal;
    }

    try{
        return stringutils::fields<T>(commaSeparatedList, ',');
    } catch (const boost::bad_lexical_cast&) {
        throw yacare::errors::BadRequest()
                << "Invalid value: '" << commaSeparatedList << "'"
                << " of parameter '" << name << "'";
    }

    return retVal;
}


template<typename Table>
typename Table::Entity loadByIdElse404(
        sql_chemistry::Gateway<Table> gtw,
        typename sql_chemistry::Gateway<Table>::Id id)
{
    if(auto e = gtw.tryLoadById(std::move(id))) { return *std::move(e); }
    throw yacare::errors::NotFound();
}


json::Value parseJsonFromRequestBodyElse400(const yacare::Request& request);

template<typename Function, typename... Args>
std::invoke_result_t<Function, Args...> tryCallElse400(Function function, Args&&... args)
try {
    return function(std::forward<Args>(args)...);
} catch (const maps::Exception& ex) {
    WARN() << "Exception caught: " << ex;
    throw yacare::errors::BadRequest() << ex.what();
}

template<typename T>
auto toJsonLocalizedValue(const T& value, const std::locale& locale)
{
    return [&](json::ObjectBuilder builder) {

        builder["value"] = value.value();
        builder["text"] = i18n::units::localize(value, locale);
    };
}

bool shouldReverseOrder(const yacare::Request& request);

template<typename Object>
std::vector<db::TId> ids(const std::vector<Object>& objects) {
    std::vector<db::TId> result;
    result.reserve(objects.size());
    for(const auto& object : objects) {
        result.push_back(object.id());
    }
    return result;
}


AssignmentReviewsMap
loadAssignmentsReviewsMap(pqxx::transaction_base& txn, const db::TIds& assignmenIds);

TasksGroupEmailsMap
loadTasksGroupEmailsMap(pqxx::transaction_base&, const db::TIds& taskGroupIds);

TasksGroupKit
loadTasksGroupKit(pqxx::transaction_base&, db::TId taskGroupId);

void setTasksGroupId(db::TId tasksGroupId, db::ugc::TasksGroupEmails&);

inline std::string baseUrl(const yacare::Request& request)
{
    std::optional<std::string> rewriteHost = request.optParam("rewrite_host");
    if (rewriteHost.has_value()) {
        return rewriteHost.value();
    }
    return "https://" + request.env("HTTP_HOST") + "/";
}

inline std::string makeFeatureImageUrl(const std::string& baseUrl, db::TId featureId)
{
    return baseUrl + "photos/" + std::to_string(featureId) + "/image";
}

inline std::string makeFeatureThumbnailUrl(const std::string& baseUrl,
                                           db::TId featureId)
{
    return baseUrl + "photos/" + std::to_string(featureId) + "/thumbnail";
}

std::optional<chrono::TimePoint>
parseOptionalDateParam(const yacare::Request& request, const std::string& param);

} // namespace tasks_planner
} // namespace mrc
} // namespace maps

YCR_QUERY_CUSTOM_PARAM(("lang"), locale, maps::Locale)
{
    std::string lang;
    bool hasLang = yacare::impl::parseArg(lang, request, "lang", true);
    if (!hasLang) {
        return false;
    }

    return boost::conversion::try_lexical_convert<maps::Locale>(lang, dest);
}

YCR_QUERY_CUSTOM_PARAM(("uid"), uid, std::string)
{
    std::string res;
    bool has = yacare::impl::parseArg(res, request, "uid", true);
    if (!has) {
        return false;
    }

    dest = res;
    return true;
}


YCR_QUERY_CUSTOM_PARAM(("login"), login, std::string)
{
    std::string res;
    bool has = yacare::impl::parseArg(res, request, "login", true);
    if (!has) {
        return false;
    }

    dest = res;
    return true;
}

YCR_QUERY_PARAM(request_id, std::string);
