#include "disjoint_polygons.h"

#include <maps/libs/common/include/unique_ptr.h>
#include <maps/libs/geolib/include/common.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>
#include <maps/libs/geolib/include/conversion_geos.h>
#include <maps/libs/introspection/include/hashing.h>
#include <maps/libs/json/include/builder.h>

#include <boost/functional/hash.hpp>
#include <geos/geom/Polygon.h>
#include <geos/operation/buffer/BufferOp.h>
#include <geos/simplify/DouglasPeuckerSimplifier.h>

#include <algorithm>
#include <memory>
#include <utility>
#include <unordered_set>

namespace maps {
namespace mrc {
namespace tasks_planner {

namespace {

constexpr double BUFFER = 1e-5;
constexpr double SIMPLIFICATION_TOLERANCE = BUFFER * 2;

using GeometryHolder = std::unique_ptr<geos::geom::Geometry>;

std::unordered_set<std::pair<size_t, size_t>, introspection::Hasher>
findIntersections(const std::vector<geolib3::Polygon2>& polygons)
{
    std::unordered_set<std::pair<size_t, size_t>, introspection::Hasher> result;

    geolib3::StaticGeometrySearcher<geolib3::Polygon2, size_t> searcher;
    for(size_t i = 0; i< polygons.size(); ++i) {
        searcher.insert(&polygons[i], i);
    }
    searcher.build();

    for(size_t i = 0; i < polygons.size(); ++i) {
        auto searchResult = searcher.find(polygons[i].boundingBox());
        for (auto it = searchResult.first; it != searchResult.second; ++it)
        {
            size_t idx = it->value();
            if (idx == i) {
                continue;
            }

            result.insert(std::minmax(idx, i));
        }
    }
    return result;
}

geolib3::Polygon2
difference(const geolib3::Polygon2& one, const geolib3::Polygon2& other)
{
    auto geosGeomOne = geolib3::internal::geolib2geosGeometry(one);
    auto geosGeomOther = geolib3::internal::geolib2geosGeometry(other);

    GeometryHolder geosGeomOtherBuffered(geosGeomOther->buffer(BUFFER));
    GeometryHolder geosNewGeom{geosGeomOne->difference(geosGeomOtherBuffered.get())};

    //intentially support only simple intersections cases
    return geolib3::Polygon2(common::dynamic_unique_cast<geos::geom::Polygon>(std::move(geosNewGeom)),
                             geolib3::Validate::Yes);
}

} // namespace


geolib3::MultiPolygon2
makeMultiPolygon(std::vector<geolib3::Polygon2> polygons)
{
    for(auto& idxPair : findIntersections(polygons)) {
        if (geolib3::spatialRelation(polygons[idxPair.first],
                                     polygons[idxPair.second],
                                     geolib3::Intersects))
        {
            polygons[idxPair.first] = difference(polygons[idxPair.first], polygons[idxPair.second]);
        }
    }

    return geolib3::simplify(
        geolib3::MultiPolygon2(std::move(polygons)),
        SIMPLIFICATION_TOLERANCE,
        geolib3::Validate::Yes);
}

} // namespace tasks_planner
} // namespace mrc
} // namespace maps
