#include "globals.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/environment.h>
#include <maps/libs/common/include/exception.h>

#include <chrono>

namespace maps {
namespace mrc {
namespace tasks_planner {

constexpr std::chrono::seconds MONGO_RETRIES_TIMEOUT_AFTER_SHUTDOWN(60);

Globals::Data::Data(common::Config cfg,
                    grinder::Config grinderConfig,
                    const std::string& roadGraphFolder,
                    const std::string& pedestrianGraphFolder,
                    const auth::TvmtoolSettings& tvm)
    : config(std::move(cfg))
    , poolHolder(config.makePoolHolder())
    , grinderLogger(grinderConfig, MONGO_RETRIES_TIMEOUT_AFTER_SHUTDOWN)
    , grinderClient(std::move(grinderConfig), grinderLogger)
    , mds(mds::Mds(config.makeMdsClient()))
    , blackbox(std::make_unique<blackbox_client::BlackboxClient>(tvm))
    , passportFrontendTvmIds(maps::common::getYandexEnvironment() == maps::common::Environment::Stable
                                 ? std::vector{PASSPORT_FRONTEND_TVM_ID_STABLE}
                                 : std::vector{PASSPORT_FRONTEND_TVM_ID_STABLE, PASSPORT_FRONTEND_TVM_ID_TESTING})
    , roadMatcher(roadGraphFolder, adapters::MATCHER_CONFIG_RESOURCE, EMappingMode::Standard)
    , pedestrianMatcher(pedestrianGraphFolder, adapters::MATCHER_CONFIG_RESOURCE, EMappingMode::Standard)
{
}

void Globals::useConfigs(common::Config config,
                         grinder::Config grinderConfig,
                         const std::string& roadGraphFolder,
                         const std::string& pedestrianGraphFolder,
                         const auth::TvmtoolSettings& tvm)
{
    INFO() << "Initializing from config";
    instance().data_.reset(new Data(std::move(config),
                                    std::move(grinderConfig),
                                    roadGraphFolder,
                                    pedestrianGraphFolder,
                                    tvm));
}

Globals& Globals::instance()
{
    static Globals instance;
    return instance;
}


const common::Config& Globals::config()
{
    REQUIRE(instance().data_, "Globals is not initialized");
    return instance().data_->config;
}

pgpool3::Pool& Globals::pool()
{
    REQUIRE(instance().data_, "Globals is not initialized");
    return instance().data_->poolHolder.pool();
}

grinder::Client& Globals::grinderClient()
{
    REQUIRE(instance().data_, "Globals is not initialized");
    return instance().data_->grinderClient;
}

mds::Mds& Globals::mds()
{
    REQUIRE(instance().data_, "Globals is not initialized");
    return instance().data_->mds;
}

blackbox_client::IBlackboxClient& Globals::blackbox()
{
    REQUIRE(instance().data_, "Globals is not initialized");
    return *instance().data_->blackbox;
}

const std::vector<auth::TvmId>& Globals::passportFrontendTvmIds()
{
    REQUIRE(instance().data_, "Globals is not initialized");
    return instance().data_->passportFrontendTvmIds;
}

adapters::CompactGraphMatcherAdapter& Globals::roadMatcher()
{
    REQUIRE(instance().data_, "Globals is not initialized");
    return instance().data_->roadMatcher;
}

adapters::CompactGraphMatcherAdapter& Globals::pedestrianMatcher()
{
    REQUIRE(instance().data_, "Globals is not initialized");
    return instance().data_->pedestrianMatcher;
}

IBlackboxClientHolder Globals::swap(IBlackboxClientHolder blackbox)
{
    ASSERT(blackbox);
    std::swap(blackbox, instance().data_->blackbox);
    return blackbox;
}

} // namespace tasks_planner
} // namespace mrc
} // namespace maps
