#include "common.h"
#include "globals.h"
#include "serialization.h"
#include "tasks_group_report.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/sql_chemistry/include/exceptions.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/serialization.h>
#include <yandex/maps/i18n.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/std/vector.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/deprecated/localeutils/include/locale.h>
#include <maps/libs/deprecated/localeutils/include/localemapper.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/infra/yacare/include/helpers.h>

#include <boost/range/adaptor/transformed.hpp>

#include <algorithm>
#include <string>
#include <vector>
#include <utility>
#include <unordered_map>

namespace maps {
namespace mrc {
namespace tasks_planner {

namespace {

const std::vector<std::pair<db::ugc::TasksGroupStatus, db::ugc::TasksGroupStatus>>
    ALLOWED_STATUS_TRANSITIONS{
        {db::ugc::TasksGroupStatus::Draft, db::ugc::TasksGroupStatus::Generating},
        {db::ugc::TasksGroupStatus::Generating, db::ugc::TasksGroupStatus::Draft},
        {db::ugc::TasksGroupStatus::Failed, db::ugc::TasksGroupStatus::Generating},
        {db::ugc::TasksGroupStatus::Open, db::ugc::TasksGroupStatus::Draft},
        {db::ugc::TasksGroupStatus::Open, db::ugc::TasksGroupStatus::InProgress},
        {db::ugc::TasksGroupStatus::InProgress, db::ugc::TasksGroupStatus::Closed},
    };


bool isStatusTransitionAllowed(db::ugc::TasksGroupStatus oldStatus, db::ugc::TasksGroupStatus newStatus)
{
    return std::find(ALLOWED_STATUS_TRANSITIONS.begin(), ALLOWED_STATUS_TRANSITIONS.end(),
                     std::make_pair(oldStatus, newStatus)) !=
                ALLOWED_STATUS_TRANSITIONS.end();
}

void startTasksGenerationOnGrinder(db::TId tasksGroupId)
{
    INFO() << "Creating grinder task for tasks group " << tasksGroupId;
    auto grinderTaskId = Globals::grinderClient()
        .submit(
            json::Value(
                {
                    {"type", json::Value("generate_tasks")},
                    {"tasks_group_id", json::Value(tasksGroupId)},
                    {"__idempotent", json::Value(true)}
                }
            )
        );
    INFO() << "Created grinder task with id " << grinderTaskId;
}


void deleteTasks(pqxx::transaction_base& txn,
                 db::TId tasksGroupId)
{
    auto taskIds = db::ugc::TaskGateway(txn).loadIds(db::ugc::table::Task::tasksGroupId == tasksGroupId);
    db::ugc::TargetGateway(txn)
        .remove(db::ugc::table::Target::taskId.in(taskIds));
    db::ugc::TaskNameGateway(txn)
        .remove(db::ugc::table::TaskName::taskId.in(taskIds));
    db::ugc::TaskGateway(txn).removeByIds(taskIds);
}

void changeTasksStatusesTo(pqxx::transaction_base& txn,
                           db::TId tasksGroupid,
                           db::ugc::TaskStatus status)
{
    db::ugc::TaskGateway gtw(txn);
    auto tasks = gtw.load(db::ugc::table::Task::tasksGroupId == tasksGroupid);
    for (auto& task : tasks) {
        task.setStatus(status);
    }
    gtw.update(tasks);
}

void changeActiveAssignmentStatuseTo(pqxx::transaction_base& txn,
                                     db::TId tasksGroupId,
                                     db::ugc::AssignmentStatus status)
{
    db::ugc::AssignmentGateway gtw(txn);
    auto assignments = gtw.load(
        db::ugc::table::Task::tasksGroupId == tasksGroupId &&
        db::ugc::table::Assignment::taskId == db::ugc::table::Task::id &&
        db::ugc::table::Assignment::status == db::ugc::AssignmentStatus::Active);
    for (auto& assignment : assignments) {
        if (status == db::ugc::AssignmentStatus::Completed) {
            assignment.markAsCompleted();
        } else if (status == db::ugc::AssignmentStatus::Revoked) {
            assignment.markAsRevoked();
        }
    }
    gtw.update(assignments);
}


bool tasksStatusesStatsRequested(const yacare::Request& request)
{
    const std::string& PARAM = "with_stats";
    return request.input().has(PARAM) &&
        request.input()[PARAM] == "true";
}

TaskStatusesMap calcTaskStatusStats(pqxx::transaction_base& txn,
                                    const db::TIds& tasksGroupsIds)
{
    TaskStatusesMap  result;

    auto tasks = db::ugc::TaskGateway(txn)
        .load(db::ugc::table::Task::tasksGroupId.in(tasksGroupsIds));

    for (const auto& task : tasks) {
        ASSERT(task.tasksGroupId());
        result[*task.tasksGroupId()][task.status()] += 1;
    }
    return result;
}

void updateEmails(pqxx::transaction_base& txn,
                  db::TId tasksGroupId,
                  db::ugc::TasksGroupEmails& newEmails)
{
    auto less = [](auto& lhs, auto& rhs) { return lhs.email() < rhs.email(); };
    auto email = [](auto& item) { return item.email(); };

    for (const auto& item : newEmails) {
        REQUIRE(tasksGroupId == item.tasksGroupId(), "invalid tasksGroupId");
    }

    auto oldEmails = db::ugc::TasksGroupEmailGateway{txn}.load(
        db::ugc::table::TasksGroupEmail::tasksGroupId == tasksGroupId);

    std::sort(newEmails.begin(), newEmails.end(), less);
    std::sort(oldEmails.begin(), oldEmails.end(), less);

    auto emailsToRemove = db::ugc::TasksGroupEmails{};
    std::set_difference(oldEmails.begin(),
                        oldEmails.end(),
                        newEmails.begin(),
                        newEmails.end(),
                        std::back_inserter(emailsToRemove),
                        less);
    auto rngToRemove = emailsToRemove | boost::adaptors::transformed(email);

    auto emailsToInsert = db::ugc::TasksGroupEmails{};
    std::set_difference(newEmails.begin(),
                        newEmails.end(),
                        oldEmails.begin(),
                        oldEmails.end(),
                        std::back_inserter(emailsToInsert),
                        less);

    db::ugc::TasksGroupEmailGateway{txn}.remove(
        db::ugc::table::TasksGroupEmail::tasksGroupId == tasksGroupId &&
        db::ugc::table::TasksGroupEmail::email.in(
            {rngToRemove.begin(), rngToRemove.end()}));
    db::ugc::TasksGroupEmailGateway{txn}.insert(emailsToInsert);
}

} // namespace

YCR_RESPOND_TO("POST /tasks_groups/", uid = std::string(), locale = RU)
{
    auto tasksGroupKit = tryCallElse400(
        [&]() {
            return createTasksGroup(json::Value::fromString(request.body()));
        });
    auto& [tasksGroup, tasksGroupEmails] = tasksGroupKit;
    if (!uid.empty()) {
        tasksGroup.setCreatedBy(uid);
    }
    auto txn = Globals::pool().masterWriteableTransaction();
    db::ugc::TasksGroupGateway gtw(*txn);
    gtw.insert(tasksGroup);

    setTasksGroupId(tasksGroup.id(), tasksGroupEmails);
    db::ugc::TasksGroupEmailGateway{*txn}.insert(tasksGroupEmails);

    txn->commit();

    response.setStatus(yacare::HTTPStatus::Created);
    response << YCR_JSON(obj) {
        toJson(obj, tasksGroupKit, {}, locale);
    };
}

YCR_RESPOND_TO("GET /tasks_groups/", locale = RU, results = 0, skip = 0)
{
    auto sqlFilter =
        sql_chemistry::FiltersCollection{sql_chemistry::op::Logical::And};

    const std::string STATUS_PARAM = "status";
    if (request.input().has(STATUS_PARAM)) {
        auto strStatusVec = vectorQueryParam<std::string>(request, STATUS_PARAM);
        std::vector<db::ugc::TasksGroupStatus> statuses;
        statuses.reserve(strStatusVec.size());
        for (const auto& statusStr : strStatusVec) {
            auto status = tryCallElse400(tasks_planner::fromString<db::ugc::TasksGroupStatus>, statusStr);
            statuses.push_back(status);
        }

        sqlFilter.add(db::ugc::table::TasksGroup::status.in(statuses));
    }

    const std::string createdBy = request.input()["created_by"];
    if (!createdBy.empty()) {
        sqlFilter.add(db::ugc::table::TasksGroup::createdBy == createdBy);
    }

    const std::string name = request.input()["name"];
    if (!name.empty()) {
        sqlFilter.add(db::ugc::table::TasksGroup::name.ilike("%" + name + "%"));
    }

    auto order = sql_chemistry::orderBy(db::ugc::table::TasksGroup::id).asc();

    if (shouldReverseOrder(request)) {
        order.desc();
    }

    if (has(results)) {
        order.limit(results);
    }

    if (has(skip)) {
        order.offset(skip);
    }

    auto txn = Globals::pool().slaveTransaction();
    db::ugc::TasksGroupGateway gtw(*txn);
    auto objects = gtw.load(sqlFilter, order);
    const auto totalObjectsCount = gtw.count(sqlFilter);
    setTotalCountHeader(response, totalObjectsCount);

    auto tasksGroupIds = ids(objects);
    auto tasksGroupEmailsMap = loadTasksGroupEmailsMap(*txn, tasksGroupIds);

    TaskStatusesMap tasksGroupToStatusesStatsMap;
    if(tasksStatusesStatsRequested(request)){
        tasksGroupToStatusesStatsMap = calcTaskStatusStats(*txn, tasksGroupIds);
    }

    response << YCR_JSON_ARRAY(arr) {
        for (auto& object : objects) {
            arr << [&](json::ObjectBuilder builder) {
                auto tasksGroupId = object.id();
                toJson(builder,
                       TasksGroupKit{.obj = std::move(object),
                                     .emails = std::move(
                                         tasksGroupEmailsMap[tasksGroupId])},
                       tasksGroupToStatusesStatsMap,
                       locale);
            };
        }
    };
}


YCR_RESPOND_TO("GET /tasks_groups/$/", locale = RU)
{
    auto id = pathnameParam<int64_t>(0);

    auto tasksGroupKit =
        loadTasksGroupKit(*Globals::pool().slaveTransaction(), id);

    TaskStatusesMap tasksGroupToStatusesStatsMap;
    if(tasksStatusesStatsRequested(request)){
        tasksGroupToStatusesStatsMap =
            calcTaskStatusStats(*Globals::pool().slaveTransaction(), {id});
    }

    response << YCR_JSON(obj) {
        toJson(obj, tasksGroupKit, tasksGroupToStatusesStatsMap, locale);
    };
}


YCR_RESPOND_TO("DELETE /tasks_groups/$/")
{
    auto id = pathnameParam<int64_t>(0);
    auto txn = Globals::pool().masterWriteableTransaction();
    db::ugc::TasksGroupGateway gtw(*txn);

    auto tasksGroup = gtw.tryLoadById(id);

    if (!tasksGroup) {
        return;
    }

    if (tasksGroup->status() != db::ugc::TasksGroupStatus::Draft) {
        throw yacare::errors::BadRequest()
        << "Only objects in status '"
        << tasks_planner::toString(db::ugc::TasksGroupStatus::Draft)
        << "' can be deleted";
    }

    gtw.removeById(id);
    txn->commit();
}


YCR_RESPOND_TO("PATCH /tasks_groups/$/", locale = RU)
{
    auto id = pathnameParam<int64_t>(0);
    auto paramsJson = parseJsonFromRequestBodyElse400(request);

    auto txn = Globals::pool().masterWriteableTransaction();
    db::ugc::TasksGroupGateway gtw(*txn);

    auto tasksGroupKit = loadTasksGroupKit(*txn, id);

    auto& [tasksGroup, tasksGroupEmails] = tasksGroupKit;
    auto oldTasksGroupStatus = tasksGroup.status();

    tryCallElse400(patchByJson, tasksGroupKit, paramsJson);

    if (oldTasksGroupStatus != tasksGroup.status()) {
        if (!isStatusTransitionAllowed(oldTasksGroupStatus, tasksGroup.status())) {
            throw yacare::errors::BadRequest()
                << "Status transition from " << tasks_planner::toString(oldTasksGroupStatus)
                << " to " << tasks_planner::toString(tasksGroup.status())
                << " is not allowed";
        }

        if (tasksGroup.status() == db::ugc::TasksGroupStatus::Generating)
        {
            startTasksGenerationOnGrinder(tasksGroup.id());
        } else if (tasksGroup.status() == db::ugc::TasksGroupStatus::Draft) {
            deleteTasks(*txn, tasksGroup.id());
        } else if (tasksGroup.status() == db::ugc::TasksGroupStatus::InProgress) {
            changeTasksStatusesTo(*txn, tasksGroup.id(), db::ugc::TaskStatus::New);
        } else if (tasksGroup.status() == db::ugc::TasksGroupStatus::Closed) {
            changeActiveAssignmentStatuseTo(*txn, tasksGroup.id(), db::ugc::AssignmentStatus::Completed);
            changeTasksStatusesTo(*txn, tasksGroup.id(), db::ugc::TaskStatus::Done);
        }
    }
    gtw.update(tasksGroup);
    updateEmails(*txn, tasksGroup.id(), tasksGroupEmails);

    txn->commit();

    response << YCR_JSON(obj) {
        toJson(obj, tasksGroupKit, {}, locale);
    };
}


YCR_RESPOND_TO("GET /tasks_groups/$/report", locale = RU, format = "csv")
{
    auto id = pathnameParam<db::TId>(0);
    if (format == "csv") {
        response["Content-Type"] = "text/csv";
        response << csvReport(id, locale);
    }
    else if (format == "json") {
        response["Content-Type"] = "application/json";
        response << jsonReport(id, locale);
    }
    else {
        throw yacare::errors::BadRequest() << "Usupported format: " << format;
    }
}


} // namespace tasks_planner
} // namespace mrc
} // namespace maps
