#include "common.h"
#include "globals.h"
#include "serialization.h"
#include "tasks_group_report.h"

#include <maps/libs/csv/include/output_stream.h>
#include <maps/libs/json/include/builder.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>
#include <yandex/maps/i18n/i18n.h>
#include <yandex/maps/i18n/units.h>
#include <maps/infra/yacare/include/yacare.h>

#include <boost/functional/hash.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/range/adaptor/map.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <sstream>
#include <unordered_map>

namespace ba = boost::adaptors;
namespace maps::mrc::tasks_planner {
namespace {

const std::string ACTUALIZATION_DATE_FIELD = "actualization_date";
const std::string ASSIGNED_TO_FIELD = "assigned_to";
const std::string ASSIGNMENT_STATUS_FIELD = "assignment_status";
const std::string CAMERA_DIRECTION_FIELD = "camera_direction";
const std::string EMPTY = "";
const std::string GOOD_COVERAGE_FIELD = "good_coverage";
const std::string GOOD_PHOTOS_FIELD = "good_photos";
const std::string PROCESSED_PHOTOS_FIELD = "processed_photos";
const std::string TASK_DISTANCE_FIELD = "task_distance";
const std::string TASK_DISTANCE_IN_METERS_FIELD = "task_distance_in_meters";
const std::string TASK_ID_FIELD = "task_id";
const std::string TASK_NAME_FIELD = "task_name";
const std::string TOTAL_COVERAGE_FIELD = "total_coverage";

auto loadTasks(db::TId tasksGroupId)
{
    auto txn = Globals::pool().slaveTransaction();
    db::ugc::TaskGateway gtw{*txn};
    auto result = gtw.load(db::ugc::table::Task::tasksGroupId == tasksGroupId,
        orderBy(db::ugc::table::Task::id));
    if (result.empty()) {
        throw yacare::errors::NotFound();
    }
    gtw.loadNames(result);
    return result;
}

class AssignmentsLoader {
    std::unordered_map<db::TId, db::ugc::Assignment> taskIdToAssignmentMap_;

public:
    template <class TasksIds>
    explicit AssignmentsLoader(const TasksIds& tasksIds)
    {
        auto txn = Globals::pool().slaveTransaction();
        auto assignments = db::ugc::AssignmentGateway{*txn}.load(
            db::ugc::table::Assignment::taskId.in(
                {std::begin(tasksIds), std::end(tasksIds)}),
            orderBy(db::ugc::table::Assignment::id).desc());
        for (const auto& assignment : assignments) {
            auto key = assignment.taskId();
            taskIdToAssignmentMap_.insert({key, std::move(assignment)});
        }
    }

    auto ids() const
    {
        return taskIdToAssignmentMap_ | ba::map_values
            | ba::transformed([](auto& item) { return item.id(); });
    }

    auto uids() const
    {
        return taskIdToAssignmentMap_ | ba::map_values
            | ba::transformed([](auto& item) {
                  return boost::lexical_cast<blackbox_client::Uid>(
                      item.assignedTo());
              });
    }

    const db::ugc::Assignment* find(const db::ugc::Task& task) const
    {
        auto it = taskIdToAssignmentMap_.find(task.id());
        return it == taskIdToAssignmentMap_.end() ? nullptr : &it->second;
    }
};

class ReviewsLoader {
    struct Key { // to hash
        db::TId assignmentId;
        db::CameraDeviation cameraDeviation;

        auto makePair() const
        {
            return std::make_pair(
                assignmentId, db::toIntegral(cameraDeviation));
        }

        friend bool operator==(const Key& lhs, const Key& rhs)
        {
            return lhs.makePair() == rhs.makePair();
        }

        friend size_t hash_value(const Key& that)
        {
            return boost::hash_value(that.makePair());
        }
    };

    std::unordered_map<Key, db::ugc::AssignmentReview, boost::hash<Key>>
        keyToReviewMap_;

public:
    template <class AssignmentsIds>
    explicit ReviewsLoader(const AssignmentsIds& assignmentsIds)
    {
        auto txn = Globals::pool().slaveTransaction();
        auto reviews =
            db::ugc::AssignmentReviewGateway {*txn}.loadWithoutGeometry(
                db::ugc::table::AssignmentReview::assignmentId.in(
                    {std::begin(assignmentsIds), std::end(assignmentsIds)}));
        for (const auto& review : reviews) {
            Key key{review.assignmentId(), review.cameraDeviation()};
            keyToReviewMap_.insert({key, std::move(review)});
        }
    }

    const db::ugc::AssignmentReview* find(
        const db::ugc::Assignment* assignment,
        db::CameraDeviation cameraDeviation)
    {
        Key key{assignment ? assignment->id() : 0, cameraDeviation};
        auto it = keyToReviewMap_.find(key);
        return it == keyToReviewMap_.end() ? nullptr : &it->second;
    }
};

class LoginsLoader {
    blackbox_client::UidToLoginMap uidToLoginMap_;

public:
    template <class Uids>
    explicit LoginsLoader(const Uids& uids)
        : uidToLoginMap_(Globals::blackbox().uidToLoginMap(
              {std::begin(uids), std::end(uids)}))
    {
    }

    const std::string& find(const db::ugc::Assignment* assignment) const
    {
        auto it = uidToLoginMap_.find(assignment
                ? boost::lexical_cast<blackbox_client::Uid>(
                      assignment->assignedTo())
                : 0);
        return it == uidToLoginMap_.end() ? EMPTY : it->second;
    }
};

const std::string& name(const db::ugc::Task& task, const Locale& locale)
{
    auto& names = task.names();
    auto it = findBestPair(names.begin(), names.end(), locale);
    return it == names.end() ? EMPTY : it->second;
}

struct Percent {
    double fraction;

    friend std::ostream& operator<<(std::ostream& os, const Percent& that)
    {
        return os << static_cast<int>(that.fraction * 100) << "%";
    }
};

} // anonymous namespace

std::string csvReport(db::TId tasksGroupId, const Locale& ydxLocale)
{
    auto stdLocale = i18n::bestLocale(ydxLocale);
    auto tasks = loadTasks(tasksGroupId);
    auto assignments = AssignmentsLoader(
        tasks | ba::transformed([](auto& task) { return task.id(); }));
    auto reviews = ReviewsLoader(assignments.ids());
    auto logins = LoginsLoader(assignments.uids());

    std::ostringstream os;
    csv::OutputStream csvWriter(os, '\t');
    csvWriter << TASK_ID_FIELD
              << TASK_NAME_FIELD
              << TASK_DISTANCE_FIELD
              << ASSIGNMENT_STATUS_FIELD
              << ASSIGNED_TO_FIELD
              << CAMERA_DIRECTION_FIELD
              << TOTAL_COVERAGE_FIELD
              << GOOD_COVERAGE_FIELD
              << PROCESSED_PHOTOS_FIELD
              << GOOD_PHOTOS_FIELD
              << ACTUALIZATION_DATE_FIELD
              << TASK_DISTANCE_IN_METERS_FIELD;
    csvWriter.endLine();

    for (const auto& task : tasks) {
        auto assignment = assignments.find(task);
        for (auto cameraDeviation : task.cameraDeviations()) {
            auto review = reviews.find(assignment, cameraDeviation);

            csvWriter << task.id() << name(task, ydxLocale)
                      << i18n::units::localize(
                             i18n::units::Distance(task.distanceInMeters()),
                             stdLocale)
                      << (assignment ? tasks_planner::toString(
                                           assignment->status())
                                     : EMPTY)
                      << logins.find(assignment)
                      << tasks_planner::toString(cameraDeviation)
                      << Percent{review ? review->coverageFraction() : 0.}
                      << Percent{review
                                 ? review->goodCoverageFraction().value_or(0.)
                                 : 0.}
                      << (review ? review->processedPhotos().value_or(0) : 0)
                      << (review ? review->goodPhotos().value_or(0) : 0)
                      << (review && review->actualizationDate()
                                 ? chrono::formatIsoDateTime(
                                       review->actualizationDate().value())
                                 : EMPTY)
                      << task.distanceInMeters();
            csvWriter.endLine();
        }
    }

    return os.str();
}

std::string jsonReport(db::TId tasksGroupId, const Locale& ydxLocale)
{
    auto tasks = loadTasks(tasksGroupId);
    auto assignments = AssignmentsLoader(
        tasks | ba::transformed([](auto& task) { return task.id(); }));
    auto reviews = ReviewsLoader(assignments.ids());
    auto logins = LoginsLoader(assignments.uids());

    json::Builder result;
    result << [&](json::ArrayBuilder bld) {
        for (const auto& task : tasks) {
            auto assignment = assignments.find(task);
            for (auto cameraDeviation : task.cameraDeviations()) {
                auto review = reviews.find(assignment, cameraDeviation);
                bld << [&](json::ObjectBuilder bld) {
                    bld[TASK_ID_FIELD] << task.id();
                    bld[TASK_NAME_FIELD] << name(task, ydxLocale);
                    bld[TASK_DISTANCE_FIELD]
                        << (task.distanceInMeters() / 1000);
                    bld[ASSIGNMENT_STATUS_FIELD]
                        << (assignment
                                ? tasks_planner::toString(assignment->status())
                                : EMPTY);
                    bld[ASSIGNED_TO_FIELD] << logins.find(assignment);
                    bld[CAMERA_DIRECTION_FIELD]
                        << tasks_planner::toString(cameraDeviation);
                    bld[TOTAL_COVERAGE_FIELD]
                        << (review ? review->coverageFraction() : 0.);
                    bld[GOOD_COVERAGE_FIELD]
                        << (review ? review->goodCoverageFraction().value_or(0.)
                                   : 0.);
                    bld[PROCESSED_PHOTOS_FIELD]
                        << (review ? review->processedPhotos().value_or(0) : 0);
                    bld[GOOD_PHOTOS_FIELD]
                        << (review ? review->goodPhotos().value_or(0) : 0);
                    bld[ACTUALIZATION_DATE_FIELD]
                        << (review && review->actualizationDate()
                                ? chrono::formatIsoDateTime(
                                      review->actualizationDate().value())
                                : EMPTY);
                };
            }
        }
    };
    return result.str();
}

} // namespace maps::mrc::tasks_planner
