#include "common.h"

#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/process/include/process.h>

#include <algorithm>
#include <fstream>
#include <map>
#include <string>
#include <unordered_set>

namespace maps {
namespace mrc {
namespace tasks_planner {
namespace tests {

std::string schemasDir()
{
    return ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/mrc/tasks-planner/";
}

std::string schemasPath()
{
    return ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/mrc/tasks-planner/swagger.json";
}


std::map<Response, std::string>
readResponseSchemasFromSwagger(const std::string& swaggerPath)
{
    const std::unordered_set<std::string> HTTP_METHODS{
        "GET", "POST", "PATCH", "DELETE", "HEAD"
    };

    std::map<Response, std::string> responseSchemas;
    auto swaggerJson = json::Value::fromFile(swaggerPath);
    ASSERT(swaggerJson.isObject());

    auto pathsJson = swaggerJson["paths"];
    ASSERT(pathsJson.isObject());

    for(const auto& path : pathsJson.fields()) {
        auto pathJson = pathsJson[path];
        ASSERT(pathJson.isObject());

        for(const auto& method : pathJson.fields()) {
            auto methodUpper = method;
            std::transform(method.begin(), method.end(), methodUpper.begin(), ::toupper);
            if(!HTTP_METHODS.count(methodUpper)) {
                continue;
            }
            auto methodJson = pathJson[method];

            ASSERT(methodJson.isObject());
            auto responses = methodJson["responses"];
            if (!responses.exists()) {
                continue;
            }

            for (const auto& status : responses.fields()) {
                auto statusJson = responses[status];
                auto schemasJson = statusJson["schema"];
                if (!schemasJson.exists()) {
                    continue;
                }

                json::Builder builder;
                builder << schemasJson;

                // INFO() << methodUpper << " " << path << " " << status;
                responseSchemas.emplace(
                    Response{methodUpper, path, std::stoi(status)},
                    builder.str()
                );
            }
        }
    }
    return responseSchemas;
}

void validateJson(const std::string& jsonStr,
                  const std::string& schema,
                  const std::string& schemaDir)
{
    std::ofstream schemaFile("schema.json");
    schemaFile << schema;
    schemaFile.flush();

    std::ofstream jsonFile("validate.json");
    jsonFile << jsonStr;
    jsonFile.flush();

    std::ostringstream cmd;
    cmd << BinaryPath("maps/tools/json-validator/json-validator")
        << " -i validate.json"
        << " -s schema.json"
        << " -d " << schemaDir;

    auto proc = process::runInteractive(
        process::Command(
            {"/bin/sh", "-c", cmd.str()}
        )
    );

    EXPECT_EQ(proc.syncWait().exitStatus(), 0)
        << "json string '" << jsonStr << "' does not correspond to schema '"
        << schema << "': " << proc.outputStream().rdbuf() << " "
        << proc.errorsStream().rdbuf();
}

} // namespace tests
} // namespace tasks_planner
} // namespace mrc
} // namespace maps
