#include "fixture.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/walk_object_gateway.h>

#include <maps/libs/geolib/include/multipolygon.h>

namespace maps::mrc::tasks_planner::tests {

using geolib3::Polygon2;
using geolib3::MultiPolygon2;
using geolib3::PointsVector;

db::Feature
Fixture::uploadTestFeatureToMds(const std::string& date)
{
    featureCount_++;
    auto mds = Playground::instance().databaseFixture.config().makeMdsClient();
    auto mdsKey = mds.post(std::to_string(featureCount_), testImageBlob()).key();
    auto feature = db::Feature("1234",
                               geolib3::Point2(0, 0),
                               geolib3::Heading(0),
                               date,
                               std::move(mdsKey),
                               db::Dataset::Agents);
    feature.setOrientation(common::ImageOrientation());

    return feature;
}

db::ugc::TasksGroup Fixture::createTasksGroup()
{
    auto txn = txnHandle();
    db::ugc::TasksGroup a{
        db::ugc::TasksGroupStatus::InProgress,
        "Moscow 2018.02",
        geolib3::convertGeodeticToMercator(
            MultiPolygon2{
                {Polygon2{PointsVector{{37.6, 55.6},
                                        {37.7, 55.6},
                                        {37.7, 55.7},
                                        {37.6, 55.6}
                            }}
                }
            }
        ),
        db::ugc::UseRouting::No,
        {1, 2, 3, 4, 5, 6, 7},
        db::ugc::UseToll::No,
        1000
    };

    db::ugc::TasksGroupGateway gtw(*txn);
    gtw.insert(a);
    txn->commit();
    return a;
}

db::ugc::Task Fixture::createTask(db::TId tasksGroupId)
{
    auto txn = txnHandle();
    db::ugc::Task t;
    t.setTasksGroupId(tasksGroupId)
        .setStatus(db::ugc::TaskStatus::New)
        .setDistanceInMeters(100500)
        .setDuration(std::chrono::hours(24))
        .setGeodeticHull(MultiPolygon2{{Polygon2{PointsVector{{37.6, 55.6},
            {37.7, 55.6},
            {37.7, 55.7},
            {37.6, 55.6}
        }}}})
        .addName(RUSSIAN, "Moscow")
        .addTarget(
            geolib3::Polyline2{geolib3::PointsVector{{0, 0}, {0, 1}, {1, 1}}},
            db::ugc::Oneway::No,
            db::ugc::Traffic::RightHand,
            db::ugc::Direction::Bidirectional)
        .addTarget(
            geolib3::Polyline2{geolib3::PointsVector{{1, 1}, {2, 1}, {3, 1}}},
            db::ugc::Oneway::No,
            db::ugc::Traffic::RightHand,
            db::ugc::Direction::Bidirectional);

    db::ugc::TaskGateway(*txn).insert(t);
    txn->commit();
    return t;
}

db::ugc::Assignment
Fixture::createAssignment(db::ugc::Task& task, db::UserId userId)
{
    auto txn = txnHandle();
    auto assignment = task.assignTo(userId);
    db::ugc::TaskGateway{*txn}.update(task);
    db::ugc::AssignmentGateway{*txn}.insert(assignment);
    txn->commit();
    return assignment;
}

db::ugc::AssignmentReview
Fixture::fillAssignmentReview(db::TId assignmentId)
{
    auto txn = txnHandle();
    db::ugc::AssignmentReview review(assignmentId);
    review.setTolokaStatus(db::ugc::TolokaStatus::InProgress)
        .setCameraDeviation(db::CameraDeviation::Front)
        .setCoverageFraction(0.5)
        .setGoodCoverageFraction(0.4)
        .setTrackDistanceInMeters(3000)
        .setTrackDuration(std::chrono::seconds(3600))
        .setActualizationDate(chrono::parseSqlDateTime("2017-10-10 16:00:00+00:00"))
        .setProcessedPhotos(10000)
        .setProcessedPoints(100000)
        .setGoodPhotos(8000)
        .setFirstShotTime(chrono::parseSqlDateTime("2017-10-02 11:15:59+00:00"))
        .setLastShotTime(chrono::parseSqlDateTime("2017-10-02 16:25:00+00:00"))
        .setCoveredGeodeticGeom({geolib3::Polyline2(geolib3::PointsVector{{1, 1}, {2, 2}})})
        .setUncoveredGeodeticGeom({geolib3::Polyline2(geolib3::PointsVector{{2, 2}, {3, 3}})})
        .setTrackGeodeticGeom({geolib3::Polyline2(geolib3::PointsVector{{3, 3}, {4, 4}})});

    db::ugc::AssignmentReviewGateway(*txn).insert(review);
    txn->commit();
    return review;
}

db::Feature Fixture::createFeature(const std::optional<chrono::TimePoint>& time)
{
    auto txn = txnHandle();
    auto baseFeature = uploadTestFeatureToMds(chrono::formatSqlDateTime(
        time.value_or(std::chrono::system_clock::now())));
    baseFeature.setSize(1920, 1080);
    db::FeatureGateway(*txn).insert(baseFeature);
    txn->commit();
    return baseFeature;
}

db::Feature
Fixture::createFeatureWithoutSize(
        const std::optional<chrono::TimePoint>& time)
{
    auto txn = txnHandle();
    auto baseFeature = uploadTestFeatureToMds(chrono::formatSqlDateTime(
        time.value_or(std::chrono::system_clock::now())));
    db::FeatureGateway(*txn).insert(baseFeature);
    txn->commit();
    return baseFeature;
}

db::Feature Fixture::createRidePhoto(
    const std::string& userId, const std::optional<chrono::TimePoint>& time)
{
    auto txn = txnHandle();
    auto result =
        uploadTestFeatureToMds(chrono::formatSqlDateTime(time.value_or(
                                   std::chrono::system_clock::now())))
            .setSize(1920, 1080)
            .setUserId(userId);
    db::FeatureGateway(*txn).insert(result);
    txn->commit();
    return result;
}

db::Feature Fixture::createWalkPhoto(
    const std::string& userId, const std::optional<chrono::TimePoint>& time)
{
    auto txn = txnHandle();

    auto walkObject = db::WalkObject(
        db::Dataset::Walks,
        "DEVICE_ID",
        time.value_or(std::chrono::system_clock::now()),
        db::WalkFeedbackType::Other,
        geolib3::Point2(1, 1),
        "comment");
    db::WalkObjectGateway{*txn}.insertx(walkObject);

    auto result =
        uploadTestFeatureToMds(chrono::formatSqlDateTime(time.value_or(
                                   std::chrono::system_clock::now())))
            .setUserId(userId)
            .setWalkObjectId(walkObject.id());
    db::FeatureGateway(*txn).insert(result);
    txn->commit();
    return result;
}

db::Feature
Fixture::createAssignmentPhoto(db::TId assignmentId,
                               std::optional<double> quality,
                               db::CameraDeviation cameraDeviation,
                               const std::string& date)
{
    auto result = uploadTestFeatureToMds(date);
    result.setCameraDeviation(cameraDeviation)
        .setSize(1920, 1080);
    if (quality) {
        result.setQuality(quality.value());
    }

    result.setAssignmentId(assignmentId);
    auto txn = txnHandle();
    db::FeatureGateway(*txn).insert(result);
    txn->commit();
    return result;
}

db::ugc::AssignmentObjects
Fixture::createAssignmentObjects(db::TId assignmentId)
{
    auto txn = txnHandle();
    db::ugc::AssignmentObjects result{
        db::ugc::AssignmentObject{
            assignmentId, chrono::parseSqlDateTime("2017-05-17 11:03:17+03"),
            geolib3::Point2(0, 1), db::ugc::AssignmentObjectType::Barrier},
        db::ugc::AssignmentObject{
            assignmentId, chrono::parseSqlDateTime("2017-05-19 11:03:17+03"),
            geolib3::Point2(2, 4), db::ugc::AssignmentObjectType::Barrier}
            .setComment("comment")};

    db::ugc::AssignmentObjectGateway(*txn).insert(result);
    txn->commit();
    return result;
}

} // namespace maps::mrc::tasks_planner::tests
