#pragma once

#include "common.h"

#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/auth/include/test_utils.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/mongo/include/init.h>
#include <maps/tools/grinder/common/include/internal_client.h>
#include <maps/wikimap/mapspro/services/mrc/tasks-planner/lib/globals.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/grinder_fixture.h>
#include <yandex/maps/mrc/unittest/local_server.h>

#include <set>
#include <string>
#include <vector>

namespace maps::mrc::tasks_planner::tests {

class Mongocxx {
public:
    Mongocxx()
    {
        mongo::init();
    }
};

class Playground : Mongocxx {
public:
    Playground()
        : grinderConfig(grinderFixture.grinderConfigPath())
        , grinderLogger(grinderConfig, std::chrono::seconds(10))
        , grinderClient(grinderConfig, grinderLogger)
    {
    }

    static Playground& instance() {
        static Playground playground;
        return playground;
    }

    unittest::WithUnittestConfig<unittest::DatabaseFixture, unittest::MdsStubFixture> databaseFixture;
    unittest::GrinderFixture grinderFixture;

    grinder::Config grinderConfig;
    grinder::MongoTaskLogger grinderLogger;
    grinder::InternalClient grinderClient;
};

const std::string DEFAULT_FEATURE_DATE_STR = "2017-10-02 11:15:59";

const std::string TEST_ROAD_GRAPH_PATH = BinaryPath("maps/data/test/graph3");

class Fixture {
public:
    Fixture() : tvmtoolRecipeHelper_("maps/libs/auth/tests/tvmtool.recipe.conf")
    {
        clearData();
        Globals::useConfigs(Playground::instance().databaseFixture.config(),
                            Playground::instance().grinderConfig,
                            TEST_ROAD_GRAPH_PATH,
                            TEST_ROAD_GRAPH_PATH,
                            tvmtoolRecipeHelper_.tvmtoolSettings());

        for (const auto& taskId : grinderClient().listTaskIds()) {
            Playground::instance().grinderLogger.deleteRecords(taskId);
            grinderClient().deleteTask(taskId);
            grinderClient().deleteTaskChangeRequests(taskId);
        }
    }

    maps::pgpool3::TransactionHandle txnHandle() {
        return Playground::instance().databaseFixture.pool().masterWriteableTransaction();
    }

    maps::pgpool3::TransactionHandle slaveTxnHandle() {
        return Playground::instance().databaseFixture.pool().slaveTransaction();
    }

    grinder::InternalClient& grinderClient() {
        return Playground::instance().grinderClient;
    }

    void clearData() {
        auto mds = Playground::instance().databaseFixture.config().makeMdsClient();
        for (const auto& feature : db::FeatureGateway{*txnHandle()}.load()) {
            mds.del(feature.mdsKey());
        }

        Playground::instance().databaseFixture.postgres().truncateTables();
    }

    static mrc::common::Blob testImageBlob() {
        static mrc::common::Blob imageBlob = maps::common::readFileToString(SRC_("test_images/photo.jpg"));
        return imageBlob;
    }

    db::ugc::TasksGroup createTasksGroup();
    db::ugc::Task createTask(db::TId tasksGroupId);

    db::Feature createFeature(
        const std::optional<chrono::TimePoint>& time = {});

    db::Feature createFeatureWithoutSize(
        const std::optional<chrono::TimePoint>& time = {});

    db::Feature createRidePhoto(
        const std::string& userId,
        const std::optional<chrono::TimePoint>& time = {});

    db::Feature createWalkPhoto(
        const std::string& userId,
        const std::optional<chrono::TimePoint>& time = {});

    db::ugc::Assignment
    createAssignment(db::ugc::Task& task, db::UserId userId);

    db::ugc::AssignmentReview
    fillAssignmentReview(db::TId assignmentId);

    db::Feature
    createAssignmentPhoto(db::TId assignmentId,
                          std::optional<double> quality,
                          db::CameraDeviation cameraDeviation = db::CameraDeviation::Front,
                          const std::string& date = DEFAULT_FEATURE_DATE_STR);

    db::ugc::AssignmentObjects
    createAssignmentObjects(db::TId assignmentId);

private:
    db::Feature uploadTestFeatureToMds(const std::string& date = DEFAULT_FEATURE_DATE_STR);

    auth::TvmtoolRecipeHelper tvmtoolRecipeHelper_;
    size_t featureCount_ = 0;
};

struct MockBlackboxClient : blackbox_client::IBlackboxClient {
    MOCK_METHOD(std::optional<blackbox_client::Uid>,
                uidByLogin,
                (const std::string&),
                (const, override));
    MOCK_METHOD(blackbox_client::UidToLoginMap,
                uidToLoginMap,
                (blackbox_client::Uids),
                (const, override));
    MOCK_METHOD(blackbox_client::UidToUserInfoMap,
                uidToUserInfoMap,
                (blackbox_client::Uids, const blackbox_client::QueryParams&),
                (const, override));
};


} // namespace maps::mrc::tasks_planner::tests
