#include "common.h"
#include "fixture.h"

#include <maps/infra/tvm2lua/ticket_parser2_lua/lib/lua_api.h>
#include <maps/infra/yacare/include/test_utils.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/http/include/http.h>
#include <maps/libs/http/include/test_utils.h>
#include <maps/libs/http/include/urlencode.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/introspection/include/stream_output.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/takeout_data_erasure_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/takeout_ongoing_job_gateway.h>

namespace maps::mrc::db {

using introspection::operator==;
using introspection::operator!=;
using introspection::operator<<;

} // namespace maps::mrc::db

namespace maps::mrc::tasks_planner::tests {

namespace {

const std::string USER_ID = "42";
const std::string JOB_ID = "job:1";
const std::string GRINDER_TASK_ID = "5f1813a816dad1580c97a08a";

void insertJob(Fixture& fixture,
               const std::string& jobId,
               const std::string& uid,
               const std::string& grinderTaskId,
               chrono::TimePoint startedAt)
{
    auto txn = fixture.txnHandle();
    db::TakeoutOngoingJobGateway gtw(*txn);
    db::TakeoutOngoingJob ongoingJob{jobId, uid, grinderTaskId, startedAt};
    gtw.insert(ongoingJob);
    txn->commit();
}

void sendTakeoutRequest(const std::string& uid, const std::string& jobId)
{
    http::MockRequest request(http::POST, http::URL("http://localhost/gdpr/takeout"));
    request.body = "uid=" + http::urlEncode(uid) + "&job_id=" + http::urlEncode(jobId);
    request.headers["X-Ya-Service-Ticket"] = "fake";
    auto response = yacare::performTestRequest(request);
    EXPECT_EQ(response.status, 200);

    auto responseSchema = readResponseSchemasFromSwagger(schemasPath());
    validateJson(response.body,
                 responseSchema.at({"POST", "/gdpr/takeout/", 200}),
                 schemasDir());
}

std::string performTakeoutStatusRequest(const std::string& userId)
{
    static const auto schema = readResponseSchemasFromSwagger(schemasPath())
                                   .at({"GET", "/1/takeout/status/", 200});
    static const auto dir = schemasDir();

    http::MockRequest req(http::GET,
                          http::URL("http://localhost/1/takeout/status/")
                              .addParam("request_id", "r1"));
    req.headers.try_emplace(yacare::tests::USER_ID_HEADER, userId);
    req.headers.try_emplace(maps::auth::SERVICE_TICKET_HEADER, "fake");
    req.headers.try_emplace(maps::tvm2::SRC_TVM_ID_HEADER,
                            std::to_string(PASSPORT_FRONTEND_TVM_ID_TESTING));
    auto resp = yacare::performTestRequest(req);
    EXPECT_EQ(resp.status, 200);

    validateJson(resp.body, schema, dir);

    auto jsonResult = json::Value::fromString(resp.body);
    EXPECT_EQ(jsonResult["status"].toString(), "ok");
    auto data = jsonResult["data"][0];
    EXPECT_EQ(data["id"].toString(), "1");
    EXPECT_EQ(data["slug"].toString(), "photos");
    return data["state"].toString();
}

} // namespace

TEST(tasks_takeout_api, test_takeout)
{
    Fixture fixture;

    sendTakeoutRequest(USER_ID, JOB_ID);

    auto txn = fixture.slaveTxnHandle();
    db::TakeoutOngoingJobGateway gtw(*txn);
    auto stored = gtw.load();
    EXPECT_EQ(stored.size(), 1u);
    EXPECT_EQ(stored[0].jobId(), JOB_ID);
    EXPECT_EQ(stored[0].uid(), USER_ID);
}

TEST(tasks_takeout_api, test_takeout_repeated_job)
{
    const chrono::TimePoint STARTED_AT
        = chrono::TimePoint::clock::now() - std::chrono::hours(1);

    Fixture fixture;

    insertJob(fixture, JOB_ID, USER_ID, GRINDER_TASK_ID, STARTED_AT);

    sendTakeoutRequest(USER_ID, JOB_ID);

    {
        auto txn = fixture.slaveTxnHandle();
        db::TakeoutOngoingJobGateway gtw(*txn);
        auto stored = gtw.load();
        EXPECT_EQ(stored.size(), 1u);
        EXPECT_EQ(stored[0].jobId(), JOB_ID);
        // New request came too soon, the job in db was not updated
        EXPECT_EQ(stored[0].grinderTaskId(), GRINDER_TASK_ID);
        EXPECT_EQ(stored[0].startedAt(), STARTED_AT);
    }
}


TEST(tasks_takeout_api, test_takeout_expired_job)
{
    const chrono::TimePoint STARTED_AT
        = chrono::parseSqlDateTime("2019-03-01 12:00:00.000+03");

    Fixture fixture;

    insertJob(fixture, JOB_ID, USER_ID, GRINDER_TASK_ID, STARTED_AT);

    sendTakeoutRequest(USER_ID, JOB_ID);

    {
        auto txn = fixture.slaveTxnHandle();
        db::TakeoutOngoingJobGateway gtw(*txn);
        auto stored = gtw.load();
        EXPECT_EQ(stored.size(), 1u);
        EXPECT_EQ(stored[0].jobId(), JOB_ID);
        EXPECT_EQ(stored[0].uid(), USER_ID);
        EXPECT_NE(stored[0].grinderTaskId(), GRINDER_TASK_ID);
        EXPECT_GT(stored[0].startedAt(), STARTED_AT);
    }
}

TEST(tasks_takeout_api, test_takeout_status)
{
    Fixture fixture;
    yacare::tests::UserIdHeaderFixture enableUserIdThroughHeader{};
    EXPECT_EQ(performTakeoutStatusRequest(USER_ID), "empty");

    auto feature = sql_chemistry::GatewayAccess<db::Feature>::construct()
                       .setSourceId("SOURCE_ID")
                       .setTimestamp(chrono::TimePoint::clock::now())
                       .setUserId(USER_ID);
    {
        auto txn = fixture.txnHandle();
        db::FeatureGateway{*txn}.insert(feature);
        txn->commit();
    }
    EXPECT_EQ(performTakeoutStatusRequest(USER_ID), "ready_to_delete");

    auto takeoutDataErasure =
        db::TakeoutDataErasure{{},  // takeout request id
                               USER_ID,
                               chrono::TimePoint::clock::now()};
    {
        auto txn = fixture.txnHandle();
        db::TakeoutDataErasureGateway{*txn}.insert(takeoutDataErasure);
        txn->commit();
    }
    EXPECT_EQ(performTakeoutStatusRequest(USER_ID), "delete_in_progress");
}

TEST(tasks_takeout_api, test_takeout_delete)
{
    static const auto schema = readResponseSchemasFromSwagger(schemasPath())
                                   .at({"POST", "/1/takeout/delete/", 200});
    static const auto dir = schemasDir();

    Fixture fixture;
    yacare::tests::UserIdHeaderFixture enableUserIdThroughHeader{};

    http::MockRequest req(http::POST,
                          http::URL("http://localhost/1/takeout/delete/")
                              .addParam("request_id", "r1"));
    req.headers.try_emplace(yacare::tests::USER_ID_HEADER, USER_ID);
    req.headers.try_emplace(maps::auth::SERVICE_TICKET_HEADER, "fake");
    req.headers.try_emplace(maps::tvm2::SRC_TVM_ID_HEADER,
                            std::to_string(PASSPORT_FRONTEND_TVM_ID_TESTING));
    req.body = R"({ "id": ["1"] })";

    auto resp = yacare::performTestRequest(req);
    EXPECT_EQ(resp.status, 200);

    validateJson(resp.body, schema, dir);

    auto jsonResult = json::Value::fromString(resp.body);
    EXPECT_EQ(jsonResult["status"].toString(), "ok");
}

} // namespace maps::mrc::tasks_planner::tests
